<?php
/**
 * Plugin Name: Geargag Advanced Shipping for WooCommerce
 * Description: Geargag Advanced Shipping plugin you can setup your own advanced rules to determine when a shipping rate should be available for the customer. This plugin will give you full control over your WooCommerce shipping rates. Within each shipping rate you can create multiple 'OR' condition groups. This means that it is very easy to setup a shipping rate for multiple user roles in one shipping rate for example.
 * Version: 1.0.0
 * Tags: woocommerce shipping, woocommerce, shipping, woocommerce free shipping, woocommerce free, woocommerce advanced free shipping, wc free shipping, wc shipping, advanced shipping, pro shipping, table rate shipping, country shipping, free shipping
 * Author: GearGag Team
 * Author URI: https://geargag.com
 * License: GPL-3.0
 * License URI: http://www.opensource.org/licenses/gpl-license.php
 * Document URI: https://plugin.geargag.com/documentation
 * Text Domain: geargag-advanced-shipping-for-woocommerce
 * Tested up to: WordPress 5.3
 * WC requires at least: 3.2.0
 * WC tested up to: 3.8.0
 */

namespace Geargag_Advanced_Shipping;

defined('WPINC') || die();

use Geargag_Advanced_Shipping\admin\Notices;
use Geargag_Advanced_Shipping\shipping\Shipping;
use Geargag_Advanced_Shipping\tools\KSES;
use Geargag_Advanced_Shipping\tools\Register_Assets;

const PLUGIN_FILE = __FILE__;
const PLUGIN_DIR = __DIR__;

final class Plugin {
	public $admin_notices;
	public $backend_assets;
	public $shipping;

	public function __construct() {
		$this->load();
		$this->init();
		$this->core();
		$this->register_assets();
		$this->boot();
	}

	public function load() {
		require PLUGIN_DIR . '/vendor/autoload.php';
	}

	public function init() {
		new KSES();

		if (is_admin()) {
			$this->admin_notices = new Notices();
			$this->admin_notices->boot();
		}
	}

	public function core() {
		if (!is_woocommerce_active()) {
			return;
		}

		$this->shipping = new Shipping();
		$this->shipping->boot();
	}

	public function register_assets() {
		$this->backend_assets = new Register_Assets($this->register_backend_assets(), 'backend');
		$this->backend_assets->boot();
	}

	public function register_backend_assets() {
		return [
			'styles' => [
				PLUGIN_SLUG . '-settings-page' => [
					'src' => get_plugin_url('assets/css/settings_page.css'),
				],
			],
			'scripts' => [
				PLUGIN_SLUG . '-settings-page' => [
					'src' => get_plugin_url('assets/js/dist/settings_page.js'),
					'deps' => ['jquery', 'jquery-form'],
					'localize_script' => [
						'settingsPage' => [
							'saveMessage' => esc_html__('Settings Saved Successfully', 'geargag-advanced-shipping-for-woocommerce'),
						],
					],
				],
			],
		];
	}

	public function boot() {
		add_action('plugin_loaded', [$this, 'load_plugin_textdomain']);
		add_action('admin_enqueue_scripts', [$this, 'enqueue_backend_assets']);
	}

	public function load_plugin_textdomain() {
		load_plugin_textdomain('geargag-advanced-shipping-for-woocommerce');
	}

	public function enqueue_backend_assets() {
		if (is_admin()) {
			wp_enqueue_style(PLUGIN_SLUG . '-settings-page');
			wp_enqueue_script(PLUGIN_SLUG . '-settings-page');
		}
	}
}

new Plugin();
