import { registerBlockType } from '@wordpress/blocks';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import { PanelBody, TextControl, ToggleControl, SelectControl } from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import React, { useState, useEffect } from 'react'; // Import useState and useEffect
import block from './block.json';
import './main.css';
import './js/player.js';
import { editorPlayer } from './js/editorPlayer.js';


const { name, attributes, title } = block;

function getYouTubeVideoId(url) {
    const standardMatch = url.match(/[?&]v=([^&]+)/);
    const shortUrlMatch = url.match(/youtu\.be\/([^?]+)/);
    return (standardMatch && standardMatch[1]) || (shortUrlMatch && shortUrlMatch[1]) || null;
}

registerBlockType(name, {
    title,
    attributes,
    icon: <div className="gdpr-youtube__icon"></div>,

    edit({ attributes, setAttributes, isSelected }) {
        const { videoID, autoplay, disableFullscreen, disableKeyboardControls, showClosedCaptions, disablePlaybar, playerRatio } = attributes;
        const blockProps = useBlockProps({
            onFocus: isSelected ? undefined : () => setAttributes({ isSelected: true }),
            onBlur: isSelected ? undefined : () => setAttributes({ isSelected: false }),
            className: `video-block${isSelected ? ' is-selected' : ''}`,
        });

        const attributesJson = JSON.stringify({
            ...attributes,
            videoID: getYouTubeVideoId(videoID),
        });

        React.useEffect(() => {
            editorPlayer();
        }, [videoID, autoplay, disableFullscreen, disableKeyboardControls, showClosedCaptions, disablePlaybar, playerRatio]);


        const videoContainerStyle = {
            paddingTop:
                playerRatio === '16:9' ? '56.25%' :
                    playerRatio === '4:3' ? '75%' :
                        playerRatio === '1:1' ? '100%' :
                            playerRatio === '3:2' ? '66.66%' :
                                '56.25%',
        };

        return (
            <div {...blockProps}>
                <InspectorControls>

                    <PanelBody>
                        <TextControl
                            label={__('YouTube Video URL', 'gdpr-extensions-com')}
                            value={videoID}
                            onChange={(newURL) => setAttributes({ videoID: newURL })}
                        />
                    </PanelBody>
                    <PanelBody title={__('General', 'gdpr-extensions-com')}>
                        <SelectControl
                            label={__('Player Ratio', 'gdpr-extensions-com')}
                            value={playerRatio}
                            options={[
                                { label: '16:9', value: '16:9' },
                                { label: '4:3', value: '4:3' },
                                { label: '1:1', value: '1:1' },
                                { label: '3:2', value: '3:2' },
                            ]}
                            onChange={(value) => setAttributes({ playerRatio: value })}
                        />
                    </PanelBody>
                    <PanelBody title={__('Settings', 'gdpr-extensions-com')}>
                        <ToggleControl
                            label={__('Autoplay', 'gdpr-extensions-com')}
                            checked={autoplay}
                            onChange={(value) => setAttributes({ autoplay: value })}
                        />

                        <ToggleControl
                            label={__('Disable Keyboard Controls', 'gdpr-extensions-com')}
                            checked={disableKeyboardControls}
                            onChange={(value) => setAttributes({ disableKeyboardControls: value })}
                        />

                        <ToggleControl
                            label={__('Show Closed Captions', 'gdpr-extensions-com')}
                            checked={showClosedCaptions}
                            onChange={(value) => setAttributes({ showClosedCaptions: value })}
                        />

                        <ToggleControl
                            label={__('Disable Playbar', 'gdpr-extensions-com')}
                            checked={disablePlaybar}
                            onChange={(value) => setAttributes({ disablePlaybar: value })}
                        />

                        <ToggleControl
                            label={__('Disable Fullscreen', 'gdpr-extensions-com')}
                            checked={disableFullscreen}
                            onChange={(value) => setAttributes({ disableFullscreen: value })}
                        />
                    </PanelBody>
                </InspectorControls>
                <div className={`gdpr-extensions-com-youtube video-placeholder${isSelected ? ' is-selected' : ''}`}>
                    {videoID ? (
                        <div className="gdpr-youtube__wrapper">
                            <div className="gdpr-youtube__video-container" style={videoContainerStyle}>
                                <div id="gdpr-youtube__player" className='gdpr-youtube__player' data-attributes={attributesJson}></div>
                                <div className="gdpr-youtube__consent-overlay">
                                    <div className="gdpr-youtube__consent-overlay__content">
                                        <div className="gdpr-youtube__consent-overlay__headline ttt" id="headline_youtube">
                                        Interaktive Karte laden
                                        </div>
                                        <div className="gdpr-youtube__consent-overlay__text" id="overlay_text_youtube">
                                            Wenn Sie die eingebettete interaktive Karte auf dieser Website nutzen möchten, werden personenbezogene Daten (IP-Adresse) an den Betreiber des Online-Kartendienstes gesendet. Daher ist es möglich, dass der Anbieter Ihre Zugriffe speichert und Ihr Verhalten analysieren kann. Wenn Sie den Button "interaktive Karte laden" anklicken, wird ein Cookie auf Ihrem Computer gesetzt, sodass die Website weiß, dass Sie dem Anzeigen der Onlinekarte in Ihrem Browser zugestimmt haben. Dieses Cookie speichert keine personenbezogenen Daten, es erkennt lediglich. Weitere Informationen erhalten Sie hier: Datenschutz
                                        </div>
                                        <a className="gdpr-youtube__consent-overlay__btn link-action-gdpr" id="btn_youtube">Interaktive Karte laden</a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    ) : (
                        <div className="gdpr-youtube__defaultLayout" style={videoContainerStyle}>
                            <div className="gdpr-youtube__logoPortion">
                                <div className="gdpr-youtube__iconWrap">
                                    <span className="dashicons dashicons-video-alt3"></span>
                                </div>
                                <h4>{__('GDPR - Youtube Video Player')}</h4>
                            </div>
                        </div>
                    )}
                </div>
            </div>
        );
    },
    save({ attributes }) {
        const { videoID, playerRatio, headlineText, bodyText, btnText } = attributes;

        const attributesJson = JSON.stringify({
            ...attributes,
            videoID: getYouTubeVideoId(videoID),
        });

        const videoContainerStyle = {
            paddingTop:
                playerRatio === '16:9' ? '56.25%' :
                    playerRatio === '4:3' ? '75%' :
                        playerRatio === '1:1' ? '100%' :
                            playerRatio === '3:2' ? '66.66%' :
                                '56.25%',
        };

        return (
            <div>
                {videoID && (
                    <div className="gdpr-youtube__wrapper gdpr-extensions-com-youtube">
                        <div className="gdpr-youtube__video-container" style={videoContainerStyle}>
                            <div id="gdpr-youtube__player" className='gdpr-youtube__player' data-attributes={attributesJson}></div>
                            <div className="gdpr-youtube__consent-overlay">
                                <div className="gdpr-youtube__consent-overlay__content">
                                    <div className="gdpr-youtube__consent-overlay__headline rrr" id="headline_youtube">
                                        Interaktive Karte laden
                                    </div>
                                    <div className="gdpr-youtube__consent-overlay__text" id="overlay_text_youtube">
                                        Wenn Sie die eingebettete interaktive Karte auf dieser Website nutzen möchten, werden personenbezogene Daten (IP-Adresse) an den Betreiber des Online-Kartendienstes gesendet. Daher ist es möglich, dass der Anbieter Ihre Zugriffe speichert und Ihr Verhalten analysieren kann. Wenn Sie den Button "interaktive Karte laden" anklicken, wird ein Cookie auf Ihrem Computer gesetzt, sodass die Website weiß, dass Sie dem Anzeigen der Onlinekarte in Ihrem Browser zugestimmt haben. Dieses Cookie speichert keine personenbezogenen Daten, es erkennt lediglich. Weitere Informationen erhalten Sie hier: Datenschutz
                                    </div>
                                    <a className="gdpr-youtube__consent-overlay__btn link-action-gdpr" id="btn_youtube">Interaktive Karte laden</a>
                                </div>
                            </div>
                        </div>
                    </div>
                )}
            </div>
        );
    },
});

