<?php
/**
 * Plugin Name: Gateway for Wallet One and Easy Digital Downloads
 * Plugin URL: https://wordpress.org/plugins/gateway-for-wallet-one-and-easy-digital-downloads/
 * Description: Adds a payment gateway for Wallet One
 * Version: 1.0
 * Author: WacoMart
 * Author URI: https://wacomart.ru
 * Text Domain: gateway-for-wallet-one-and-easy-digital-downloads
 * Domain Path: /languages/
 * WP requires at least: 4.4
 * License: GPLv2 or later
 */
/*
Copyright 2020 WacoMart (email : info@wacomart.ru)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EDD_Wallet_One_Gateway
 */
class EDD_Wallet_One_Gateway
{

    /**
     * EDD_Wallet_One_Gateway constructor.
     */
    public function __construct()
    {

        if ( !function_exists( 'edd_is_gateway_active' ) ) {
            return;
        }

        add_action( 'edd_wallet_one_cc_form', '__return_false' );
        add_action( 'edd_gateway_wallet_one', array( $this, 'process_payment' ) );

        // Load translation
        add_action( 'init', array( $this, 'edd_wallet_one_lang' ) );

        add_action( 'init', array( $this, 'process_wo_notifications' ) );
        add_action( 'init', array( $this, 'process_wo_purchase_confirmation_url' ) );

        add_filter( 'edd_payment_gateways', array( $this, 'register_gateway' ) );
        add_filter( 'edd_accepted_payment_icons', array( $this, 'payment_icon' ) );
        add_filter( 'edd_payment_confirm_wallet_one', array( $this, 'pending_success_page' ) );
        add_filter( 'edd_settings_sections_gateways', array( $this, 'section' ), 10, 1 );
        add_filter( 'edd_settings_gateways', array( $this, 'settings' ) );

    }

    /**
     *Download translation
     *
     *
     * @since 1.0
     */
    public function edd_wallet_one_lang()
    {
        load_plugin_textdomain( 'gateway-for-wallet-one-and-easy-digital-downloads', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
    }

    /**
     *Add Gateway subsection
     *
     * @param $sections
     * @return mixed
     *
     * @since 1.0
     */
    public function section( $sections )
    {
        $sections['wallet_one'] = __( 'Wallet One', 'gateway-for-wallet-one-and-easy-digital-downloads' );

        return $sections;
    }

    /**
     *Register the gateway
     *
     * @param $gateways
     * @return mixed
     *
     * @since 1.0
     */
    public function register_gateway( $gateways )
    {
        $gateways['wallet_one'] = array(
            'admin_label' => __( 'Wallet One', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
            'checkout_label' => edd_get_option( 'wmi_wallet_one_label', __( 'Wallet One (bank cards, e-money, etc.)', 'gateway-for-wallet-one-and-easy-digital-downloads' ) ),
        );

        return $gateways;
    }

    /**
     *Register the gateway icon
     *
     * @param $icons
     * @return mixed
     *
     * @since 1.0
     */
    public function payment_icon( $icons )
    {
        $icons[ plugin_dir_url( __FILE__ ) . 'wmi_icon.png' ] = 'Wallet One';
        return $icons;
    }

    /**
     *Process the purchase data and send to WalletOne
     *
     * @param $purchase_data
     *
     * @since 1.0
     */
    public function process_payment( $purchase_data )
    {

        $credentials = $this->get_api_credentials();

        if ( empty( $credentials['wmi_merchant_id'] ) || empty( $credentials['wmi_secret_key'] ) ) {
            edd_set_error( 0, __( 'You must enter your Shop ID and Secret key in settings', 'gateway-for-wallet-one-and-easy-digital-downloads' ) );
            edd_send_back_to_checkout( '?payment-mode=' . $purchase_data['post_data']['edd-gateway'] );
        }

        if ( !$this->check_allowed_currency() ) {
            edd_set_error( 0, __( 'This currency is not supported. Available currencies: Russian rubles, US dollars, euros, Ukrainian hryvnia, South African rand, Kazakhstani tenge, Belorussian rubles, Tajik somoni, Polish zloty.', 'gateway-for-wallet-one-and-easy-digital-downloads' ) );
            edd_send_back_to_checkout( '?payment-mode=' . $purchase_data['post_data']['edd-gateway'] );
        }

        $payment_data = array(
            'price' => $purchase_data['price'],
            'date' => $purchase_data['date'],
            'user_email' => $purchase_data['user_email'],
            'purchase_key' => $purchase_data['purchase_key'],
            'currency' => edd_get_currency(),
            'downloads' => $purchase_data['downloads'],
            'cart_details' => $purchase_data['cart_details'],
            'user_info' => $purchase_data['user_info'],
            'status' => 'pending'
        );

        // record the pending payment
        $payment = edd_insert_payment( $payment_data );

        if ( $payment ) {

            $return_url = add_query_arg( array(
                'payment-confirmation' => 'wallet_one',
                'payment-id' => $payment
            ), edd_get_success_page_uri() );


            $args = [];
            $args['WMI_MERCHANT_ID'] = $credentials['wmi_merchant_id'];

            $args['WMI_PAYMENT_AMOUNT'] = number_format( ( float )$purchase_data['price'], 2, '.', '' );

            $args['WMI_CURRENCY_ID'] = $this->get_code_currency();

            $args['WMI_PAYMENT_NO'] = $payment;

            $args['WMI_DESCRIPTION'] = "BASE64:" . base64_encode( sprintf( __( 'Payment ID: %s for %s', 'gateway-for-wallet-one-and-easy-digital-downloads' ), $payment, $purchase_data['user_email'] ) );


            $args['WMI_SUCCESS_URL'] = $return_url;
            $args['WMI_FAIL_URL'] = edd_get_failed_transaction_uri( '?payment-id=' . $payment );

            $args['WMI_CUSTOMER_FIRSTNAME'] = $purchase_data['user_info']['first_name'];

            $args['WMI_CUSTOMER_LASTNAME'] = $purchase_data['user_info']['last_name'];

            $args['WMI_RECIPIENT_LOGIN'] = $purchase_data['user_email'];
            $args['WMI_CUSTOMER_EMAIL'] = $purchase_data['user_email'];

            $args['WMI_CULTURE_ID'] = $this->wo_check_locale_shop();

            $args['WMI_AUTO_LOCATION'] = '1';

            //FZ-54
            if ( edd_get_option( 'wmi_federal_law' ) ) {

                $args['WMI_CUSTOMER_EMAIL'] = $purchase_data['user_email'];

                $orderItems = array();

                foreach ( $purchase_data['cart_details'] as $item ) {
                    $item_amount = round( ( $item['subtotal'] / $item['quantity'] ) - ( $item['discount'] / $item['quantity'] ), 2 );

                    $SubTotal = round( $item_amount * $item['quantity'], 2 );

                    $orderItems[] = array(
                        'Title' => mb_strimwidth( stripslashes_deep( html_entity_decode( wp_strip_all_tags( $item['name'] ), ENT_COMPAT, 'UTF-8' ) ), '0', 128, ' ...' ),
                        'Quantity' => $item['quantity'],
                        'SubTotal' => $SubTotal,
                        'UnitPrice' => $item_amount,
                        'TaxType' => edd_get_option( 'wmi_tax_type' ),
                        'Tax' => number_format( ( float )$this->wmi_tax_formula( $SubTotal ), 2, '.', '' ),
                    );
                }

                $args['WMI_ORDER_ITEMS'] = json_encode( $orderItems, JSON_UNESCAPED_UNICODE );
            }

            $wo_secret_key = $credentials['wmi_secret_key'];

            $args['WMI_SIGNATURE'] = $this->WoSignFormation( $args, $wo_secret_key );


            $args = apply_filters( 'edd_wallet_one_redirect_args', $args );

            edd_debug_log( 'WalletOne arguments: ' . print_r( $args, 1 ) );

            $url = 'https://wl.walletone.com/checkout/checkout/Index/';

            $redirect_text = __( 'Redirecting to Wallet One site, click on button if not redirected.', 'gateway-for-wallet-one-and-easy-digital-downloads' );

            ?>
            <div class="edd-gateway-wallet-one-form" style="padding:20px;font-family:arial,sans-serif;text-align:center;color:#555">
                <h3><?php echo $redirect_text; ?></h3>
                <form id="edd_gateway_wallet_one_form" name="edd_gateway_wallet_one_form" method="post" action="<?php echo $url; ?>" accept-charset="UTF-8">
                    <?php foreach ( $args as $arg => $value ) { ?>
                        <?php if ( is_array( $value ) ) { ?>
                            <?php foreach ( $value as $val ) { ?>
                                <input type="hidden" name="<?php echo esc_attr( $arg ); ?>" value="<?php echo htmlspecialchars( $val ); ?>">
                            <?php } ?>
                        <?php } else { ?>
                            <input type="hidden" name="<?php echo esc_attr( $arg ); ?>" value="<?php echo htmlspecialchars( $value ); ?>">
                        <?php } ?>
                    <?php } ?>
                    <input type="submit" value="<?php esc_attr_e( 'Pay via Wallet One', 'gateway-for-wallet-one-and-easy-digital-downloads' ); ?>">
                </form>
                <div class="loader">
                    <img width="100px;" src="<?php echo plugin_dir_url( __FILE__ ) . 'assets/loading_spinner.gif'; ?>" alt="loader">
                </div>
                <script type="text/javascript">document.getElementById('edd_gateway_wallet_one_form').submit();</script>
            </div>
            <?php

        } else {

            edd_record_gateway_error( __( 'Wallet One Error', 'gateway-for-wallet-one-and-easy-digital-downloads' ), sprintf( __( 'Payment creation failed while processing a Wallet One payment gateway purchase. Payment data: %s', 'gateway-for-wallet-one-and-easy-digital-downloads' ), json_encode( $purchase_data ) ) );
            // if errors are present, send the user back to the purchase page so they can be corrected
            edd_send_back_to_checkout( '?payment-mode=' . $purchase_data['post_data']['edd-gateway'] );
        }
    }

    /**
     *Calculating the amount of tax
     *
     * @param $price
     * @return float|string
     *
     * @since 1.0
     */
    public function wmi_tax_formula( $price )
    {

        $wmi_tax_type = edd_get_option( 'wmi_tax_type' );

        switch ( $wmi_tax_type ) {
            case 'tax_ru_1':
            case 'tax_ru_2':
                $amount_item = round( $price * 0 / 100, 2 );
                break;
            case 'tax_ru_3':
                $amount_item = round( $price * 10 / 100, 2 );
                break;
            case 'tax_ru_4':
                $amount_item = round( $price * 18 / 100, 2 );
                break;
            case 'tax_ru_5':
                $amount_item = round( $price * 10 / 110, 2 );
                break;
            case 'tax_ru_6':
                $amount_item = round( $price * 18 / 118, 2 );
                break;
            case 'tax_ru_7':
                $amount_item = round( $price * 20 / 100, 2 );
                break;
            case 'tax_ru_8':
                $amount_item = round( $price * 20 / 120, 2 );
                break;
            default:
                $amount_item = '0.00';
        }

        return $amount_item;
    }

    /**
     *Shows "Purchase Processing" message for WalletOne payments are still pending on site return.
     *This helps address the Race Condition, as detailed in issue #1839
     *
     * @param $content
     * @return false|string
     *
     * @since 1.0
     */
    public function pending_success_page( $content )
    {

        if ( !isset( $_GET['payment-id'] ) && !edd_get_purchase_session() ) {
            return $content;
        }

        $payment_id = isset( $_GET['payment-id'] ) ? absint( $_GET['payment-id'] ) : false;

        if ( !$payment_id ) {
            $session = edd_get_purchase_session();
            $payment_id = edd_get_purchase_id_by_key( $session['purchase_key'] );
        }

        edd_empty_cart();

        $payment = get_post( $payment_id );

        if ( $payment && 'pending' == $payment->post_status ) {

            // Payment is still pending so show processing indicator to fix the Race Condition, issue #
            ob_start();

            edd_get_template_part( 'payment', 'processing' );

            $content = ob_get_clean();

        }

        return $content;

    }

    /**
     *Register the gateway settings
     *
     * @param $settings
     * @return array
     *
     * @since 1.0
     */
    public function settings( $settings )
    {

        $wallet_one_settings = array(
            array(
                'id' => 'wmi_settings',
                'name' => '<strong>' . __( 'Wallet One Gateway Settings', 'gateway-for-wallet-one-and-easy-digital-downloads' ) . '</strong>',
                'desc' => __( 'Configure your Wallet One Gateway Settings', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'type' => 'header'
            ),
            array(
                'id' => 'wmi_merchant_id',
                'name' => __( 'Shop ID', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'desc' => __( 'Enter your Wallet One Shop ID', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'type' => 'text',
            ),
            array(
                'id' => 'wmi_secret_key',
                'name' => __( 'Secret key', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'desc' => __( 'Enter your Wallet One Secret key', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'type' => 'text',
            ),
            array(
                'id' => 'wmi_url_addresses',
                'type' => 'descriptive_text',
                'name' => __( 'Data to send the results of transaction', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'desc' => '<p>' . __( 'Enter the following URL in your personal account on the Wallet One website:', 'gateway-for-wallet-one-and-easy-digital-downloads' ) . '</p>' .
                    '<p><strong><code>' . sprintf( __( 'URL script: %s', 'gateway-for-wallet-one-and-easy-digital-downloads' ), site_url( '/?edd-wallet-one=result' ) ) .
                    '</code></strong></p>'
            ),
            array(
                'id' => 'wmi_wallet_one_label',
                'name' => __( 'Checkout Label', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'desc' => __( 'This text will be shown on the checkout page when choosing a payment method', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'type' => 'text',
                'size' => 'regular'
            ),
            array(
                'id' => 'wmi_federal_law',
                'name' => __( '54 Federal law', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'desc' => __( 'Transmit details for receipts to WalletOne (Federal Law 54-FZ)', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'type' => 'checkbox',
            ),
            array(
                'id' => 'wmi_tax_type',
                'name' => __( 'VAT rate', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'desc' => __( 'Select the default VAT rate', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
                'type' => 'select',
                'options' => $this->get_tax_type(),
                'std' => '1',
            ),
        );

        if ( version_compare( EDD_VERSION, 2.5, '>=' ) ) {
            $wallet_one_settings = array( 'wallet_one' => $wallet_one_settings );
        }

        return array_merge( $settings, $wallet_one_settings );
    }

    /**
     *Process notifications sent from WalletOne
     *
     *
     * @since 1.0
     */
    public function process_wo_notifications()
    {

        if ( isset( $_REQUEST['edd-wallet-one'] ) && $_SERVER['REQUEST_METHOD'] == 'POST' ) {

            if ( $_REQUEST['edd-wallet-one'] === 'result' ) {

                edd_debug_log( 'wallet_one_payment_notifications: start' );
                edd_debug_log( print_r( sanitize_post( $_REQUEST ), 1 ) );

                $wo_response = sanitize_post( $_POST, 'raw' );

                $order_id = (int)$wo_response['WMI_PAYMENT_NO'];

                if ( empty( $order_id ) ) {
                    $error = sprintf( __( 'Error! Payment not found.', 'gateway-for-wallet-one-and-easy-digital-downloads' ) );

                    die( 'WMI_RESULT=RETRY&WMI_DESCRIPTION=' . urlencode( $error ) );
                }

                $payment = new EDD_Payment( $order_id );


                if ( $payment->gateway != 'wallet_one' ) {
                    return;
                }

                if ( $payment->status == 'publish' ) {
                    return;
                }

                $key = edd_get_option( 'wmi_secret_key' );

                $wo_sign = $this->WoSignFormation( $wo_response, $key );


                if ( $wo_response['WMI_SIGNATURE'] == $wo_sign ) {

                    if ( strtoupper( $wo_response['WMI_ORDER_STATE'] ) == 'ACCEPTED' ) {

                        $order_validation = $this->validate_order_payment( $wo_response, $payment );

                        if ( false === $order_validation['success'] ) {
                            edd_update_payment_status( $payment->ID, 'failed' );
                            edd_insert_payment_note( $payment->ID, $order_validation['message'] );
                        } else {

                            edd_insert_payment_note( $payment->ID, __( 'The order has been successfully paid in Wallet One.', 'gateway-for-wallet-one-and-easy-digital-downloads' ) );
                            edd_set_payment_transaction_id( $payment->ID, $wo_response['WMI_ORDER_ID'] );
                            edd_update_payment_status( $payment->ID, 'publish' );
                        }

                    }

                    die( 'WMI_RESULT=OK' );
                } else {

                    $error = sprintf( __( 'Validate hash error. Local: %1$s Remote: %2$s', 'gateway-for-wallet-one-and-easy-digital-downloads' ), $wo_sign, $wo_response['WMI_SIGNATURE'] );
                    edd_insert_payment_note( $payment->ID, $error . $order_id );

                    die( 'WMI_RESULT=RETRY&WMI_DESCRIPTION=' . urlencode( $error ) );
                }
            }

            edd_debug_log( 'wallet_one_payment_notifications: error, action not found' );
            wp_die( __( 'Api request error. Action not found.', 'gateway-for-wallet-one-and-easy-digital-downloads' ), 'Payment error', array( 'response' => '503' ) );
        }

    }

    /**
     *When customers are redirected back to the EDD site from Wallet One,
     *we need to set up the purchase session to match their URL variables
     *
     * @since 1.0
     */
    public function process_wo_purchase_confirmation_url()
    {

        if ( isset( $_GET['payment-id'] ) && isset( $_GET['payment-confirmation'] ) && 'wallet_one' == $_GET['payment-confirmation'] ) {

            $payment_id = sanitize_text_field( $_GET['payment-id'] );

            $payment = new EDD_Payment( $payment_id );

            // Set up the EDD purchase session. This makes the edd_receipt shortcode know what to show
            edd_set_purchase_session( $purchase_data = array(
                'purchase_key' => $payment->key
            ) );
        }
    }

    /**
     *Choose a language: English or Russian
     *
     * @return string
     *
     * @since 1.0
     */
    public function wo_check_locale_shop()
    {
        $locale = get_locale();

        if ( $locale == 'ru_RU' ) {
            $locale = 'ru-RU';
        } else {
            $locale = 'en-US';
        }
        return $locale;
    }

    /**
     *Get tax type
     *
     * @return array
     *
     * @since 1.0
     */
    public function get_tax_type()
    {
        return array(
            'tax_ru_1' => __( 'Without VAT', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
            'tax_ru_2' => __( '0%', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
            'tax_ru_3' => __( '10%', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
            'tax_ru_4' => __( '18%', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
            'tax_ru_5' => __( 'Settlement rate 10/110', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
            'tax_ru_6' => __( 'Settlement rate 18/118', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
            'tax_ru_7' => __( '20%', 'gateway-for-wallet-one-and-easy-digital-downloads' ),
            'tax_ru_8' => __( 'Settlement rate 20/120', 'gateway-for-wallet-one-and-easy-digital-downloads' )
        );
    }

    /**
     *Checking the payment amount
     *
     * @param $data
     * @param $payment
     * @return array
     *
     * @since 1.0
     */
    private function validate_order_payment( $data, $payment )
    {
        $success = true;
        $message = '';

        if ( $payment->total > number_format( $data['WMI_PAYMENT_AMOUNT'], 2, '.', '' ) ) {
            $success = false;
            $message = sprintf( __( 'Wallet One total (%s) did not match payment total (%s).', 'gateway-for-wallet-one-and-easy-digital-downloads' ), $data['WMI_PAYMENT_AMOUNT'], $payment->total );
        }

        return array( 'success' => $success, 'message' => $message );
    }

    /**
     *Check store currency
     *
     * @return bool
     *
     * @since 1.0
     */
    private function check_allowed_currency()
    {

        $ret = false;
        $currency = edd_get_currency();

        switch ( $currency ) {

            case 'RUB' :
            case 'USD' :
            case 'EUR' :
            case 'UAH' :
            case 'ZAR' :
            case 'KZT':
            case 'BYR':
            case 'TJS':
            case 'PLN':

                $ret = true;
                break;

        }

        return $ret;
    }

    /**
     *Currency code
     *
     * @return string
     *
     * @since 1.0
     */
    private function get_code_currency()
    {

        $currency = edd_get_currency();

        switch ( $currency ) {
            case 'RUB' :
                $currency_code = '643';
                break;
            case 'USD' :
                $currency_code = '840';
                break;
            case 'EUR' :
                $currency_code = '978';
                break;
            case 'UAH' :
                $currency_code = '980';
                break;
            case 'ZAR' :
                $currency_code = '710';
                break;
            case 'KZT':
                $currency_code = '398';
                break;
            case 'BYR':
                $currency_code = '974';
                break;
            case 'TJS':
                $currency_code = '972';
                break;
            case 'PLN':
                $currency_code = '985';
                break;
            default:
                $currency_code = '643';
        }

        return $currency_code;
    }

    /**
     *Retrieve the API credentials
     *
     * @return null[]
     *
     * @since 1.0
     */
    private function get_api_credentials()
    {
        global $edd_options;

        $wmi_merchant_id = isset( $edd_options['wmi_merchant_id'] ) ? trim( $edd_options['wmi_merchant_id'] ) : null;
        $wmi_secret_key = isset( $edd_options['wmi_secret_key'] ) ? trim( $edd_options['wmi_secret_key'] ) : null;

        return $data = array(
            'wmi_merchant_id' => $wmi_merchant_id,
            'wmi_secret_key' => $wmi_secret_key,
        );
    }

    /**
     *Hash calculation
     *
     * @param $data
     * @param $secret_key
     * @return string
     *
     * @since 1.0
     */
    private function WoSignFormation( $data, $secret_key )
    {
        if ( !empty( $data['WMI_SIGNATURE'] ) ) unset( $data['WMI_SIGNATURE'] );

        foreach ( $data as $name => $val ) {
            if ( is_array( $val ) ) {
                usort( $val, "strcasecmp" );
                $data[ $name ] = $val;
            }
        }

        uksort( $data, "strcasecmp" );
        $fieldValues = "";

        foreach ( $data as $key => $value ) {
            if ( !preg_match( '/WMI_/', $key ) ) continue;

            if ( is_array( $value ) )
                foreach ( $value as $v ) {
                    $v = iconv( "utf-8", "windows-1251", $v );
                    $fieldValues .= wp_unslash( $v );
                }
            else {
                $value = iconv( "utf-8", "windows-1251", $value );
                $fieldValues .= wp_unslash( $value );
            }
        }

        return base64_encode( pack( "H*", md5( $fieldValues . $secret_key ) ) );
    }

}

/**
 *Load our plugin
 *
 *
 * @since 1.0
 */
function edd_wallet_one_load()
{
    $gateway = new EDD_Wallet_One_Gateway;
    unset( $gateway );
}

add_action( 'plugins_loaded', 'edd_wallet_one_load' );
