<?php
/*
Plugin Name: Gateway de Pagamento Bitcoin Simplificado
Plugin URI:
Description: Plugin simples para aceitar pagamentos em Bitcoin, com endereço, QR code e cotação personalizáveis.
Version: 1.0
Author: MuriloLivex
Author URI:
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: gateway-de-pagamento-bitcoin-simplificado
Domain Path: /languages/
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Adiciona o CSS para o layout responsivo apenas nas páginas necessárias.
 */
function gbs_enqueue_gateway_styles() {
	if ( is_checkout() || is_wc_endpoint_url( 'order-received' ) ) {
		$css = "
		.gbs-payment-instructions {
			max-width: 100% !important;
			box-sizing: border-box;
		}
		.gbs-payment-inner {
			display: flex;
			flex-wrap: wrap;
			align-items: flex-start;
			gap: 16px;
		}
		.gbs-btc-qr {
			flex: 0 0 150px; /* Base fixa de 150px, sem encolher */
			max-width: 150px;
		}
		.gbs-btc-content {
			flex: 1 1 200px; /* Pode crescer e encolher a partir de 200px */
			min-width: 0; /* Permite que o flex item encolha abaixo do seu conteúdo */
			max-width: 100%;
		}
		.gbs-btc-address {
			font-family: monospace;
			font-size: 1.0em;
			word-break: break-all; /* Força a quebra de linha do endereço */
			overflow-wrap: break-word;
			max-width: 100%;
			padding: 8px;
			background-color: #f5f5f5;
			border: 1px solid #ddd;
			border-radius: 4px;
			margin: 8px 0;
			box-sizing: border-box;
		}
		@media (max-width: 500px) {
			.gbs-payment-inner {
				flex-direction: column; /* Coloca os itens em coluna */
			}
			.gbs-btc-qr {
				margin-bottom: 15px;
			}
			.gbs-btc-address {
				font-size: 0.9em;
			}
		}
		";
		wp_add_inline_style( 'woocommerce-inline', $css );
		
		// Adiciona o JavaScript para o botão copiar
		$js = "
		document.addEventListener('click', function(e) {
			var copyBtn = e.target.closest('.gbs-copy-btn');
			if (copyBtn) {
				e.preventDefault();
				var originalText = copyBtn.textContent;
				var addressToCopy = copyBtn.getAttribute('data-address');
				if (navigator.clipboard) {
					navigator.clipboard.writeText(addressToCopy).then(function() {
						copyBtn.textContent = 'Copiado!';
						setTimeout(function() {
							copyBtn.textContent = originalText;
						}, 2000);
					});
				}
			}
		});
		";
		wp_add_inline_script( 'jquery', $js );
	}
}
add_action( 'wp_enqueue_scripts', 'gbs_enqueue_gateway_styles' );

add_action( 'plugins_loaded', 'gw_btc_simplificado_init', 11 );
function gw_btc_simplificado_init() {
	if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
		return;
	}

	class GW_BTC_Simplificado_Gateway extends WC_Payment_Gateway {
		public function __construct() {
			$this->id                 = 'gw_btc_simplificado';
			$this->icon               = '';
			$this->has_fields         = true;
			$this->method_title       = __( 'Gateway de Pagamento Bitcoin Simplificado', 'gateway-de-pagamento-bitcoin-simplificado' );
			$this->method_description = __( 'Aceite pagamentos em Bitcoin com configurações personalizáveis.', 'gateway-de-pagamento-bitcoin-simplificado' );

			$this->init_form_fields();
			$this->init_settings();

			$this->title        = $this->get_option( 'title' );
			$this->description  = $this->get_option( 'description' );
			$this->btc_address  = $this->get_option( 'btc_address' );
			$this->qr_image_url = $this->get_option( 'qr_image_url' );
			$this->btc_price    = floatval( $this->get_option( 'btc_price' ) );
			$this->instruction  = $this->get_option( 'instruction' );

			add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
			add_action( 'woocommerce_thankyou_' . $this->id, array( $this, 'pagina_obrigado' ) );
		}

		public function init_form_fields() {
			$this->form_fields = array(
				'enabled' => array(
					'title'   => __( 'Enable/Disable', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'type'    => 'checkbox',
					'label'   => __( 'Enable Bitcoin Gateway', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'default' => 'yes'
				),
				'title' => array(
					'title'       => __( 'Title', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'type'        => 'text',
					'description' => __( 'This controls the title which the user sees during checkout.', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'default'     => __( 'Bitcoin Payment', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'desc_tip'    => true,
				),
				'description' => array(
					'title'       => __( 'Description', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'type'        => 'textarea',
					'description' => __( 'Payment method description that the customer will see on your checkout.', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'default'     => __( 'Pay with Bitcoin directly to our wallet.', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'desc_tip'    => true,
				),
				'btc_address' => array(
					'title'       => __( 'Bitcoin Address', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'type'        => 'text',
					'description' => __( 'Your Bitcoin wallet address', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'default'     => '',
					'desc_tip'    => true,
				),
				'qr_image_url' => array(
					'title'       => __( 'QR Code Image URL', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'type'        => 'text',
					'description' => __( 'URL of the QR code image for your Bitcoin address', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'default'     => '',
					'desc_tip'    => true,
				),
				'btc_price' => array(
					'title'       => __( 'Bitcoin Price in Local Currency', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'type'        => 'number',
					'description' => __( 'Current Bitcoin price in your store currency', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'default'     => '150000',
					'desc_tip'    => true,
				),
				'instruction' => array(
					'title'       => __( 'Instructions', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'type'        => 'textarea',
					'description' => __( 'Instructions for the customer', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'default'     => __( 'Send the exact BTC amount to the address above and your order will be processed.', 'gateway-de-pagamento-bitcoin-simplificado' ),
					'desc_tip'    => true,
				),
			);
		}

		public function process_payment( $order_id ) {
			$order = wc_get_order( $order_id );
			$order->update_status( 'on-hold', __( 'Aguardando pagamento via Bitcoin', 'gateway-de-pagamento-bitcoin-simplificado' ) );
			wc_reduce_stock_levels( $order_id );
			WC()->cart->empty_cart();

			return array(
				'result'   => 'success',
				'redirect' => $order->get_checkout_order_received_url(),
			);
		}

		public function pagina_obrigado( $order_id ) {
			$allowed_html = array(
				'div' => array( 'class' => array(), 'style' => array() ),
				'p' => array( 'class' => array(), 'style' => array() ),
				'strong' => array(),
				'img' => array( 'src' => array(), 'alt' => array(), 'style' => array() ),
				'button' => array( 'type' => array(), 'class' => array(), 'data-address' => array(), 'style' => array() ),
				'br' => array()
			);
			echo wp_kses( $this->mostrar_instrucoes_pagamento( $order_id, true ), $allowed_html );
		}

		public function payment_fields() {
			if ( $this->description ) {
				echo wp_kses_post( wpautop( $this->description ) );
			}
			$allowed_html = array(
				'div' => array( 'class' => array(), 'style' => array() ),
				'p' => array( 'class' => array(), 'style' => array() ),
				'strong' => array(),
				'img' => array( 'src' => array(), 'alt' => array(), 'style' => array() ),
				'button' => array( 'type' => array(), 'class' => array(), 'data-address' => array(), 'style' => array() ),
				'br' => array()
			);
			echo wp_kses( $this->mostrar_instrucoes_pagamento( null, false ), $allowed_html );
		}

		private function mostrar_instrucoes_pagamento( $order_id = null, $com_botao_copiar = false ) {
			$btc_address  = $this->get_option( 'btc_address' );
			$qr_image_url = $this->get_option( 'qr_image_url' );
			$btc_price    = $this->get_option( 'btc_price' ) ?: 150000;
			$instruction  = $this->get_option( 'instruction' );

			if ( $order_id ) {
				$order = wc_get_order( $order_id );
				$total = $order ? $order->get_total() : 0;
			} else {
				$total = WC()->cart ? WC()->cart->total : 0;
			}

			$btc_amount = $btc_price > 0 ? $total / $btc_price : 0;
			$btc_amount = number_format( $btc_amount, 8, '.', '' );

			ob_start();
			?>
			<div class="gbs-payment-instructions" style="border:1px solid #ccc; padding:16px; border-radius:12px; background:#fff; max-width:100%; margin-top: 1em; box-sizing: border-box;">
				<div class="gbs-payment-inner">
					<?php if ( $qr_image_url ) : ?>
						<div class="gbs-btc-qr">
							<img src="<?php echo esc_url( $qr_image_url ); ?>" alt="<?php esc_attr_e( 'Código QR Bitcoin', 'gateway-de-pagamento-bitcoin-simplificado' ); ?>" style="max-width:100%; height:auto; border-radius:8px;">
						</div>
					<?php endif; ?>

					<div class="gbs-btc-content">
						<p><strong><?php esc_html_e( 'Endereço Bitcoin:', 'gateway-de-pagamento-bitcoin-simplificado' ); ?></strong></p>
						<p class="gbs-btc-address"><?php echo esc_html( $btc_address ); ?></p>

						<?php if ( $com_botao_copiar ) : ?>
							<button type="button" class="gbs-copy-btn" data-address="<?php echo esc_attr( $btc_address ); ?>" style="background:#333; color:#fff; padding:6px 20px; border:none; border-radius:4px; cursor:pointer;">
								<?php esc_html_e( 'Copiar endereço', 'gateway-de-pagamento-bitcoin-simplificado' ); ?>
							</button>
							<br><br>
						<?php endif; ?>

						<p><strong><?php esc_html_e( 'Total do pedido em BTC:', 'gateway-de-pagamento-bitcoin-simplificado' ); ?></strong></p>
						<p><?php echo esc_html( $btc_amount ); ?> BTC</p>

						<p style="color:#666; font-size:0.9em;"><?php echo esc_html( $instruction ); ?></p>
					</div>
				</div>
			</div>
			<?php
			return ob_get_clean();
		}
	}

	add_filter(
		'woocommerce_payment_gateways',
		function ( $gateways ) {
			$gateways[] = 'GW_BTC_Simplificado_Gateway';
			return $gateways;
		}
	);
}