<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class GWCPA_Frontend {

    public function __construct() {
        add_filter( 'woocommerce_coupon_is_valid', [ $this, 'validate_coupon_on_apply' ], 20, 2 );
        add_action( 'woocommerce_before_calculate_totals', [ $this, 'revalidate_and_remove_on_update' ], 100 );
        add_filter( 'woocommerce_coupon_get_discount_amount', [ $this, 'apply_discount_cap' ], 10, 5 );
        add_action( 'woocommerce_review_order_before_payment', [ $this, 'display_banner_location_1' ] );
        add_action( 'woocommerce_before_checkout_form', [ $this, 'display_banner_location_2' ], 5 );
        add_action( 'woocommerce_before_cart_totals', [ $this, 'display_banner_location_3' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ] );
        add_action( 'wp_ajax_gwcpa_apply_coupon', [ $this, 'ajax_apply_coupon' ] );
        add_action( 'wp_ajax_nopriv_gwcpa_apply_coupon', [ $this, 'ajax_apply_coupon' ] );
    }
    
    public function ajax_apply_coupon() {
        check_ajax_referer( 'gwcpa-nonce', 'security' );
        if ( ! function_exists('WC') || empty($_POST['coupon_code']) ) {
            wp_send_json_error();
        }
        $coupon_code = sanitize_text_field( wp_unslash( $_POST['coupon_code'] ) );
        if ( WC()->cart->apply_coupon( $coupon_code ) ) {
            wp_send_json_success();
        } else {
            wp_send_json_error(); 
        }
        wp_die();
    }

    public function validate_coupon_on_apply( $is_valid, $coupon ) {
        if ( ! $is_valid ) return $is_valid;
        $required_gateways = $coupon->get_meta( GWCPA_META_KEY );
        if ( empty( $required_gateways ) ) return true;
        $chosen_gateway = WC()->session->get( 'chosen_payment_method' );
        if ( ! empty( $chosen_gateway ) && ! in_array( $chosen_gateway, (array) $required_gateways ) ) {
            throw new Exception( esc_html( $this->get_validation_error_message( $coupon, false ) ) );
        }
        return true;
    }

    public function revalidate_and_remove_on_update( $cart ) {
        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) return;
        if ( did_action( 'woocommerce_before_calculate_totals' ) >= 2 ) return;
        $applied_coupons = $cart->get_applied_coupons();
        if ( empty( $applied_coupons ) ) return;
        $chosen_gateway = WC()->session->get( 'chosen_payment_method' );
        if ( empty( $chosen_gateway ) ) return;
        foreach ( $applied_coupons as $coupon_code ) {
            $coupon = new WC_Coupon( $coupon_code );
            $required_gateways = $coupon->get_meta( GWCPA_META_KEY );
            if ( ! empty( $required_gateways ) && ! in_array( $chosen_gateway, (array) $required_gateways ) ) {
                $cart->remove_coupon( $coupon_code );
                $this->add_validation_notice($coupon);
            }
        }
    }
    
    public function add_validation_notice($coupon) {
        $error_message = $this->get_validation_error_message($coupon);
        if ( ! wc_has_notice( $error_message, 'error' ) ) {
            wc_add_notice( $error_message, 'error' );
        }
    }
    
    public function get_validation_error_message($coupon, $with_html = true) {
        $settings = get_option(GWCPA_OPTION_NAME, []);
        $default_error = __( 'Sorry, the coupon code {coupon_code} is only valid for payment with {gateway_names}.', 'gateway-coupon-assistant' );
        $error_template = !empty($settings['validation_error_message']) ? $settings['validation_error_message'] : $default_error;
        $required_gateways = $coupon->get_meta(GWCPA_META_KEY);
        $gateways_db = WC()->payment_gateways->payment_gateways();
        $gateway_names = [];
        foreach ( (array) $required_gateways as $gateway_id ) {
            if(isset($gateways_db[$gateway_id])) $gateway_names[] = $gateways_db[$gateway_id]->get_title();
        }
        $separator = ' ' . _x('or', 'gateway separator', 'gateway-coupon-assistant') . ' ';
        $gateway_names_str = implode( $separator, $gateway_names );
        if ($with_html) {
            $replacements = ['{coupon_code}'   => '<code>' . esc_html($coupon->get_code()) . '</code>', '{gateway_names}' => '<strong>' . esc_html($gateway_names_str) . '</strong>'];
        } else {
            $replacements = ['{coupon_code}'   => esc_html($coupon->get_code()), '{gateway_names}' => esc_html($gateway_names_str)];
        }
        return str_replace(array_keys($replacements), array_values($replacements), $error_template );
    }

    public function apply_discount_cap( $discount, $discounting_amount, $cart_item, $single, $coupon ) {
        if ( ! $coupon->is_type('percent') ) {
            return $discount;
        }
        $discount_cap = $coupon->get_meta('_gwcpa_discount_cap');
        if ( is_numeric( $discount_cap ) && $discount_cap > 0 && $discount > $discount_cap ) {
            return (float) $discount_cap; 
        }
        return $discount; 
    }

    public function display_banner_location_1() {
        $settings = get_option( GWCPA_OPTION_NAME, [] ); 
        if ( empty( $settings['location_1_enabled'] ) ) return;
        $visibility = isset($settings['location_1_visibility']) ? $settings['location_1_visibility'] : 'all';
        if ( ( $visibility === 'desktop' && wp_is_mobile() ) || ( $visibility === 'mobile' && !wp_is_mobile() ) ) return;
        $this->render_banner_html($settings);
    }

    public function display_banner_location_2() {
        $settings = get_option( GWCPA_OPTION_NAME, [] ); 
        if ( empty( $settings['location_2_enabled'] ) ) return;
        $visibility = isset($settings['location_2_visibility']) ? $settings['location_2_visibility'] : 'all';
        if ( ( $visibility === 'desktop' && wp_is_mobile() ) || ( $visibility === 'mobile' && !wp_is_mobile() ) ) return;
        if (is_checkout()) { $this->render_banner_html($settings); }
    }

    public function display_banner_location_3() {
        $settings = get_option( GWCPA_OPTION_NAME, [] ); 
        if ( empty( $settings['location_3_enabled'] ) ) return;
        $visibility = isset($settings['location_3_visibility']) ? $settings['location_3_visibility'] : 'all';
        if ( ( $visibility === 'desktop' && wp_is_mobile() ) || ( $visibility === 'mobile' && !wp_is_mobile() ) ) return;
        if (is_cart()) { $this->render_banner_html($settings); }
    }
    
    public function render_banner_html($settings) {
        if ( empty( $settings['featured_coupon_code'] ) ) return;
        $coupon_code = sanitize_text_field($settings['featured_coupon_code']);
        $coupon = new WC_Coupon( $coupon_code );
        if ( ! $coupon->get_id() ) return;

        $coupon_amount = ( $coupon->get_discount_type() === 'percent' ) ? $coupon->get_amount() . '%' : wc_price( $coupon->get_amount() );
        $required_gateways = $coupon->get_meta( GWCPA_META_KEY );
        $gateway_names = '';
        if ( !empty($required_gateways) && is_array($required_gateways) ) {
            $gateways_db = WC()->payment_gateways->payment_gateways(); $names = [];
            foreach ($required_gateways as $id) { if (isset($gateways_db[$id])) $names[] = $gateways_db[$id]->get_title(); }
            if (!empty($names)) {
                $separator = ' ' . _x('or', 'gateway separator', 'gateway-coupon-assistant') . ' ';
                $gateway_names = implode($separator, $names);
            }
        }
        
        $banner_message_template = isset($settings['banner_message']) ? $settings['banner_message'] : '';
        $template = isset($settings['banner_template']) ? sanitize_key($settings['banner_template']) : 'style1';
        $primary_color = isset($settings['primary_color']) ? sanitize_hex_color($settings['primary_color']) : '#26a69a';
        $text_color = isset($settings['text_color']) ? sanitize_hex_color($settings['text_color']) : '#ffffff';
        $accent_color = isset($settings['accent_color']) ? sanitize_hex_color($settings['accent_color']) : '#f2bf1a';

        if ($template === 'style3') {
            $coupon_code_html = sprintf( "<a href='#' class='gwcpa-apply-coupon-trigger' data-coupon-code='%s' style='background-color: %s; color: %s; padding: 2px 8px; margin: 0 5px; border-radius: 4px; cursor: pointer; display: inline-block; text-decoration: none;'>%s</a>", esc_attr($coupon_code), esc_attr($text_color), esc_attr($primary_color), esc_html($coupon_code));
        } else {
            $coupon_code_html = sprintf( "<a href='#' class='gwcpa-apply-coupon-trigger' data-coupon-code='%s' style='cursor: pointer; font-weight: bold; border: 1px dashed; padding: 2px 6px; border-radius: 4px; text-decoration: none;'>%s</a>", esc_attr($coupon_code), esc_html($coupon_code));
        }
        
        $gateway_names_styled = ($template === 'style3') ? sprintf( "<strong style='color: %s;'>%s</strong>", esc_attr($accent_color), esc_html($gateway_names)) : '<strong>'.esc_html($gateway_names).'</strong>';
        
        $message = str_replace( 
            ['{gateway_names}', '{coupon_amount}', '{coupon_code}'], 
            [$gateway_names_styled, '<strong>'.wp_kses_post($coupon_amount).'</strong>', $coupon_code_html], 
            $banner_message_template 
        );
        
        echo '<div class="gwcpa-banner-wrapper" style="margin: 20px 0; clear:both;">';
        if ($template === 'style1') {
            printf( '<div style="background-color: %s; color: %s; padding: 12px 20px; border-radius: 8px; font-size: 15px;">%s</div>', esc_attr($primary_color), esc_attr($text_color), wp_kses_post($message) );
        }
        if ($template === 'style2') {
            printf( '<div style="border: 2px solid %s; border-radius: 8px; padding: 20px; text-align: center; background-color: #f7f7f7;"><p style="font-size: 16px; margin: 0; color: #333;">%s</p></div>', esc_attr($primary_color), wp_kses_post($message) );
        }
        if ($template === 'style3') {
             printf( '<div style="background-color: %s; color: %s; padding: 10px 15px; border-radius: 6px; font-size: 14px; line-height: 1.8; text-align: center;">%s</div>', esc_attr($primary_color), esc_attr($text_color), wp_kses_post($message) );
        }
        echo '</div>';
    }

    public function enqueue_assets() {
        if ( is_cart() || is_checkout() ) {
            $script_handle = 'gwcpa-auto-apply-js'; 
            wp_enqueue_script( $script_handle, GWCPA_PLUGIN_URL . 'assets/js/gwcpa-auto-apply.js', ['jquery', 'wc-cart-fragments'], GWCPA_VERSION, true );
            wp_localize_script( $script_handle, 'gwcpa_ajax_vars', array(
                'ajax_url'      => admin_url( 'admin-ajax.php' ),
                'nonce'         => wp_create_nonce( 'gwcpa-nonce' ),
                'applying_text' => esc_html__( 'Applying...', 'gateway-coupon-assistant' ),
                'applied_text'  => esc_html__( 'Applied!', 'gateway-coupon-assistant' ),
                'error_text'    => esc_html__( 'Error!', 'gateway-coupon-assistant' ),
            ));
        }
        if ( is_checkout() ) {
            wp_enqueue_script('gwcpa-checkout-validation-js', GWCPA_PLUGIN_URL . 'assets/js/checkout-validation.js', ['jquery'], GWCPA_VERSION, true);
        }
    }
}