<?php
/**
 * Filters
 *
 * @package GamiPress\LearnPress\Points_Per_Quiz_Score\Filters
 * @since 1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Quiz completed listener
 *
 * @param int $quiz_id
 * @param int $course_id
 * @param int $user_id
 * @param WP_User $current_user
 */
function gamipress_points_per_quiz_score_for_learnpress_quiz_completed( $quiz_id, $course_id, $user_id ) {

    // Start with an underscore to hide fields from custom fields list
    $prefix = '_gamipress_points_per_quiz_score_for_learnpress_';

    
    $user = learn_press_get_user( $user_id );

    $user_quiz = $user->get_item_data( $quiz_id, $course_id );

    $quiz = $user_quiz->get_quiz();

    $quiz_title = $quiz->get_heading_title();

    $quiz_results = $user_quiz->calculate_results();

    $user_mark = $quiz_results['user_mark'];
    $status = $quiz_results['status'];
    

    foreach( gamipress_get_points_types() as $points_type => $data ) {

        // Skip if award points is not checked
        if( ! (bool) gamipress_get_post_meta( $data['ID'], $prefix . 'award_points' ) ) {
            continue;
        }

        $percent = (int) gamipress_get_post_meta( $data['ID'], $prefix . 'percent' );

        // Skip if percent is not higher than 0
        if( $percent <= 0 ) {
            continue;
        }

        // Check if award is limited to 1 time only
        if( (bool) gamipress_get_post_meta( $data['ID'], $prefix . 'one_time' ) ) {
            // Skip if points of this type has been already awarded
            if( (bool) get_user_meta( $user_id, $prefix . $points_type . '_awarded_' . $quiz_id, true ) ) {
                continue;
            }
        }

        // Check if only looking for passed quizzes
        if( (bool) gamipress_get_post_meta( $data['ID'], $prefix . 'only_passed' ) ) {
            // Skip if user has not passed the quiz
            if( $status !== "completed" ) {
                continue;
            }
        }

        // Setup the ratio value used to convert the score into points
        $ratio = $percent / 100;

        $points_to_award = absint( $user_mark * $ratio );

        /**
         * Filter to allow override this amount at any time
         *
         * @since 1.0.0
         *
         * @param int       $points_to_award    Points amount that will be awarded
         * @param int       $user_id            User ID that will receive the points
         * @param string    $points_type        Points type slug of the points amount
         * @param int       $percent            Percent setup on the points type
         * @param int       $quiz_id            Quiz ID
         *
         * @return int
         */
        $points_to_award = (int) apply_filters( 'gamipress_points_per_quiz_score_for_learnpress_points_to_award', $points_to_award, $user_id, $points_type, $percent, $quiz_id );

        // Award the points to the user
        if( $points_to_award > 0 ) {
            
            gamipress_award_points_to_user( $user_id, $points_to_award, $points_type );

            // Insert the custom user earning for the manual balance adjustment
            gamipress_insert_user_earning( $user_id, array(
                'title'	        => sprintf(
                    /* translators: %1$s: Points to award %2$s: Quiz Name %3$s: Score */
                    __( '%1$s for complete the quiz "%2$s" with a score of %3$s', 'gamipress-points-per-quiz-score-for-learnpress' ),
                    gamipress_format_points( $points_to_award, $points_type ),
                    $quiz_title,
                    $user_mark . ''
                ),
                'user_id'	    => $user_id,
                'post_id'	    => $data['ID'],
                'post_type' 	=> 'points-type',
                'points'	    => $points_to_award,
                'points_type'	=> $points_type,
                'date'	        => gmdate( 'Y-m-d H:i:s', current_time( 'timestamp' ) ),
            ) );

            // Set a user meta to meet that points have been awarded
            update_user_meta( $user_id, $prefix . $points_type . '_awarded_' . $quiz_id, '1' );

        }
    }
}
$quiz_submitted_hook = ( defined( 'LEARNPRESS_VERSION' ) && version_compare( LEARNPRESS_VERSION, '3.0', '>=' ) ? 'learn-press/user/quiz-finished' : 'learn_press_user_finish_quiz' );
add_action( $quiz_submitted_hook, 'gamipress_points_per_quiz_score_for_learnpress_quiz_completed', 10, 3 );