/**
 * WordPress dependencies
 */
import { createBlock, pasteHandler } from '@wordpress/blocks';
import { BlockControls } from '@wordpress/block-editor';
import { DropdownMenu, ToolbarGroup } from '@wordpress/components';
import { createHigherOrderComponent } from '@wordpress/compose';
import { dispatch } from '@wordpress/data';
import { Fragment, useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';

/**
 * Internal dependencies
 */
import API from '../modules/api';
import FuzionIcon from '../../../img/icon.svg';
import { showError } from '../gutenberg/notice';
import CodeEditOptions from '../gutenberg/modals/code-edit';

const allowedBlocks = [
	'core/code',
	'core/paragraph'
];

/**
 * Add a dropdown menu to the block toolbar.
 *
 * @since 1.0.0
 */
export const withDropdownControls = createHigherOrderComponent( ( BlockEdit ) => {
	/**
	 * Render component.
	 *
	 * @param {Object} props Component props.
	 */
	return ( props ) => {
		// If current block is not allowed.
		if ( ! allowedBlocks.includes( props.name ) ) {
			return (
				<BlockEdit { ...props } />
			);
		}

		const { attributes, setAttributes } = props;
		const [ credits, setCredits ] = useState( 300 );
		const [ instructions, setInstructions ] = useState( '' );
		const [ processing, setProcessing ] = useState( false );
		const [ replace, setReplace ] = useState( false );
		const [ showModal, setShowModal ] = useState( false );

		/**
		 * Options for core/paragraph block.
		 */
		const paragraphOptions = [
			{
				icon: 'editor-expand',
				title: __( 'Expand text', 'fuzion' ),
				onClick: () => performAiQuery( { prompt: attributes.content, type: 'expand' } ),
			},
			{
				icon: 'editor-spellcheck',
				title: __( 'Grammar correction', 'fuzion' ),
				onClick: () => performAiQuery( { prompt: attributes.content, type: 'grammar' } ),
			},
			{
				icon: 'editor-alignleft',
				title: __( 'TL;DR summarization', 'fuzion' ),
				onClick: () => performAiQuery( { prompt: attributes.content, type: 'tldr' }, false ),
			},
		];

		/**
		 * Options for core/code block.
		 *
		 * @since 1.4.0
		 */
		const codeOptions = [
			{
				icon: 'editor-code',
				title: __( 'Generate code (beta)', 'fuzion' ),
				onClick: () => performAiQuery( { prompt: attributes.content, type: 'code_generate' } ),
			},
			{
				icon: 'editor-code',
				title: __( 'Edit code (beta)', 'fuzion' ),
				onClick: () => setShowModal( true ),
			},
		];

		const generateCode = () => {
			setProcessing( true );

			const data = {
				instructions,
				credits,
				prompt: attributes.content,
				type: 'code_edit'
			};

			performAiQuery( data, replace );
		};

		/**
		 * Perform API query.
		 *
		 * @param {Object}  data                API query object.
		 * @param {string}  [data.instructions] Instructions for code generation. Only used for code edits.
		 * @param {number}  [data.credits]      Credits to use for generation. Only used for code edits.
		 * @param {string}  data.type           Request type.
		 * @param {string}  data.prompt         Prompt to use.
		 * @param {boolean} replaceBlock        Replace current content.
		 */
		const performAiQuery = ( data, replaceBlock = true ) => {
			dispatch( 'core/notices' ).removeNotice( 'fuzion-error' );

			const currentClasses = props.attributes.className;
			setAttributes( { ...attributes, className: currentClasses + ' fuzion-loading' } );

			const api = new API;
			const block = props.name;
			const insertBlocksAfter = props.insertBlocksAfter;
			const endpoint = 'core/paragraph' === block ? 'ai_query' : 'ai_code';

			api.post( endpoint, data )
				.then( ( response ) => {
					if ( ! response.success ) {
						showError( response.data );
						return;
					}

					// code_edit === data.type

					if ( 'core/code' === block && 'code_generate' === data.type ) {
						const content = pasteHandler( {
							plainText: response.data.message[ 0 ],
							mode: 'AUTO',
						} );

						insertBlocksAfter( content );
					} else if ( replaceBlock ) {
						setAttributes( { ...attributes, content: response.data.message[ 0 ] } );
					} else {
						const blockType = 'code_edit' === data.type ? 'core/code' : 'core/paragraph';
						const content = createBlock(
							blockType,
							{ content: response.data.message[ 0 ] }
						);

						dispatch( 'core/block-editor' ).insertBlocks( content );
					}
				} )
				.catch( ( error ) => {
					showError( error.data );
				} )
				.finally( () => {
					setInstructions( '' );
					setShowModal( false );
					setProcessing( false );
					setAttributes( { ...attributes, className: currentClasses } );
				} );
		};

		return (
			<Fragment>
				<BlockEdit { ...props } />
				<BlockControls>
					<ToolbarGroup>
						<DropdownMenu
							icon={ <FuzionIcon width="20px" height="20px" /> }
							label={ __( 'Fuzion AI options', 'fuzion' ) }
							controls={
								'core/paragraph' === props.name ? paragraphOptions : codeOptions
							}
						/>
					</ToolbarGroup>
				</BlockControls>
				{ showModal && <CodeEditOptions
					cancelCb={ () => setShowModal( false ) }
					credits={ credits }
					generate={ generateCode }
					instructions={ instructions }
					processing={ processing }
					replace={ replace }
					setCredits={ setCredits }
					setInstructions={ setInstructions }
					setReplace={ setReplace }
				/> }
			</Fragment>
		);
	};
}, 'withDropdownControl' );
