/**
 * WordPress dependencies
 */
import { createBlock, createBlocksFromInnerBlocksTemplate, pasteHandler } from '@wordpress/blocks';
import { TabPanel } from '@wordpress/components';
import { dispatch } from '@wordpress/data';
import { PluginSidebar } from '@wordpress/edit-post';
import { useEffect, useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import { doAction } from '@wordpress/hooks';

/**
 * Internal dependencies
 */
import AiContentModal from '../gutenberg/modals/content-results';
import GeneralPanel from '../gutenberg/panels/general';
import ImagesPanel from '../gutenberg/panels/images';
import monthlyVolumeChart from '../charts/volume';
import SEOPanel from '../gutenberg/panels/seo';
import { mobileVsDesktopChart } from '../charts/mobile-vs-desktop';
import { percentageChart } from '../charts/percentage';
import { showError } from '../gutenberg/notice';
import { tabs } from '../misc/tabs';

/**
 * Register the plugin sidebar in the Gutenberg Editor via the Plugin Sidebar API.
 *
 * This sidebar allows selecting of various predefined actions for generating content based on user input.
 *
 * @since 1.0.0
 * @param {Object} props
 * @param {Object} props.api
 */
const AIPluginSidebar = ( { api } ) => {
	const [ type, setType ] = useState( 'expand' );
	const [ prompt, setPrompt ] = useState( '' );
	const [ response, setResponse ] = useState( [] );
	const [ checked, setChecked ] = useState( [] );
	const [ processing, setProcessing ] = useState( false );

	// SEO data.
	const [ keyword, setKeyword ] = useState( '' );
	const [ seoData, setSeoData ] = useState( {} );

	// Content AI settings.
	const [ credits, setCredits ] = useState( 80 );
	const [ temperature, setTemperature ] = useState( 60 );
	const [ results, setResults ] = useState( 1 );

	// Modal controls.
	const [ showModal, setShowModal ] = useState( false );

	// Image AI.
	const [ imagePrompt, setImagePrompt ] = useState( '' );

	useEffect( () => {
		if ( ! hasSeoData() ) {
			return;
		}

		setProcessing( false );
		drawCharts();
	}, [ seoData ] );

	useEffect( () => {
		if ( checked.length === 1 && 'code_generate' === type ) {
			addToEditor();
		}
	}, [ checked, response ] );

	/**
	 * Has seo data saved in component state.
	 *
	 * @return {boolean} True if there is data, false otherwise.
	 */
	const hasSeoData = () => {
		return Object.entries( seoData ).length > 0;
	};

	/**
	 * Process generation on "Generate" button click.
	 */
	const processGenerate = () => {
		initProcess();

		const endpoint = 'code_generate' === type ? 'ai_code' : 'ai_query';

		api.post( endpoint, { credits, prompt, results, temperature, type } )
			.then( ( res ) => {
				if ( ! res.success ) {
					cancelGenerate();
					showError( res.data );
					return;
				}

				setProcessing( false );
				setResponse( res.data.message );

				// If there's only one element - check it, because the modal won't have a checkbox.
				if ( res.data.message.length === 1 ) {
					setChecked( [ 0 ] );
				}

				if ( 'code_generate' !== type ) {
					setShowModal( true );
				}
			} )
			.catch( ( error ) => {
				cancelGenerate();
				showError( error.data );
			} );
	};

	/**
	 * Calculate how much to shift the percentage label.
	 *
	 * @param {number} percent
	 * @return {string} CSS percent value.
	 */
	const difficultyPercentageForDiv = ( percent ) => {
		const value = ( Math.floor( 0.39 * percent ) * 2.5 ) + 0.83333;
		return value + '%';
	};

	/**
	 * Draw charts.
	 */
	const drawCharts = () => {
		if ( ! hasSeoData() ) {
			return;
		}

		new monthlyVolumeChart( seoData.searchVolume );
		mobileVsDesktopChart( seoData.percentMobileSearches, seoData.percentDesktopSearches );
		percentageChart( seoData.clickedAnyResult, 'clickedAnyResult' );
		percentageChart( seoData.percentPaidClicksChartValue, 'paidClicks' );
		percentageChart( seoData.rankingDifficulty / 100, 'difficulty' );
		document.getElementById( 'rankingDifficultySpan' ).style.left = difficultyPercentageForDiv( seoData.rankingDifficulty );
	};

	/**
	 * Init generation process.
	 */
	const initProcess = () => {
		dispatch( 'core/notices' ).removeNotice( 'fuzion-error' );
		setProcessing( true );
	};

	/**
	 * Process action on "Analyze keyword" button click.
	 */
	const processAnalyze = () => {
		initProcess();
		api.post( 'seo_query', { keyword } )
			.then( ( res ) => {
				if ( ! res.success ) {
					cancelAnalyze();
					showError( res.data );
					return;
				}

				setSeoData( res.data );
			} )
			.catch( ( error ) => {
				cancelAnalyze();
				showError( error.data );
			} );
	};

	/**
	 * Generate image.
	 *
	 * @since 1.3.0
	 */
	const generateImage = () => {
		initProcess();
		const postID = wp.data.select( 'core/editor' ).getCurrentPostId();

		api.post( 'ai_image', { imagePrompt, postID } )
			.then( ( res ) => {
				if ( ! res.success ) {
					cancelGenerateImage();
					showError( res.data );
					return;
				}

				addImageToEditor( res.data.id, res.data.url );
			} )
			.catch( ( error ) => {
				cancelGenerateImage();
				showError( error.data );
			} );
	};

	/**
	 * Stop generating from "Generate content" modal.
	 */
	const cancelGenerate = () => {
		setProcessing( false );
		setResponse( [] );
		setChecked( [] );
		setPrompt( '' );
		setShowModal( false );
	};

	/**
	 * Stop SEO analysis.
	 */
	const cancelAnalyze = () => {
		setProcessing( false );
		setKeyword( '' );
	};

	/**
	 * Stop image generation.
	 */
	const cancelGenerateImage = () => {
		setProcessing( false );
		setImagePrompt( '' );
	};

	/**
	 * Create block based content
	 *
	 * @since 1.4.0
	 *
	 * @param {string} content
	 * @return {Object} Block content.
	 */
	const createBlockContent = ( content ) => {
		if ( type === 'list' ) {
			const isOrderedList = /^\d+\. /.test( content.trim() );
			const pattern = isOrderedList ? '\\d+\\. ' : content.at( 0 ) + ' ';

			const list = content.replace( new RegExp( pattern, 'g' ), '' ).split( '\n' ).filter( ( li ) => !! li );
			const innerBlocksTemplate = list.map( ( li ) => {
				return [ 'core/list-item', { content: li } ];
			} );

			return createBlock( 'core/list', { ordered: isOrderedList }, createBlocksFromInnerBlocksTemplate( innerBlocksTemplate ) );
		}

		// Remove quotes from titles and headings.
		if ( type === 'title' || type === 'heading' ) {
			content = content.replace( /(^"|"$)/g, '' );
		}

		if ( type === 'title' ) {
			dispatch( 'core/editor' ).editPost( { title: content } );
			return;
		}

		if ( type === 'code_generate' ) {
			const codeBlock = pasteHandler( {
				plainText: content,
				mode: 'AUTO',
			} );

			return createBlocksFromInnerBlocksTemplate( codeBlock );
		}

		return createBlock( type === 'heading' ? 'core/heading' : 'core/paragraph', { content } );
	};

	/**
	 * Add content to Gutenberg editor from the "Generate content" modal.
	 */
	const addToEditor = () => {
		checked.forEach( ( index ) => {
			const content = createBlockContent( response[ index ] );

			if ( content ) {
				dispatch( 'core/block-editor' ).insertBlocks( content );
			}
		} );

		cancelGenerate();

		doAction( 'aiContent.complete' );
	};

	/**
	 * Add generated image to editor.
	 *
	 * @since 1.3.0
	 *
	 * @param {number} id  Attachment ID.
	 * @param {string} url Attachment URL.
	 */
	const addImageToEditor = ( id, url ) => {
		const content = createBlock(
			'core/image',
			{ id, sizeSlug: 'large', url }
		);

		dispatch( 'core/block-editor' ).insertBlocks( content );
		cancelGenerateImage();
	};

	/**
	 * Set the content variant checked state.
	 *
	 * @param {string} key Variant key.
	 */
	const selectItem = ( key ) => {
		if ( checked.includes( key ) ) {
			setChecked( checked.filter( ( item ) => item !== key ) );
		} else {
			setChecked( [ ...checked, key ] );
		}
	};

	/**
	 * Render tab.
	 *
	 * @param {Object} tab
	 * @return {JSX.Element} Selected tab content.
	 */
	const renderTab = ( tab ) => {
		if ( 'content' === tab.name ) {
			return <GeneralPanel
				credits={ credits }
				processGenerate={ processGenerate }
				processing={ processing }
				prompt={ prompt }
				results={ results }
				setCredits={ setCredits }
				setPrompt={ setPrompt }
				setResults={ setResults }
				setTemperature={ setTemperature }
				setType={ setType }
				temperature={ temperature }
				type={ type }
			/>;
		}

		if ( 'images' === tab.name ) {
			return <ImagesPanel
				imagePrompt={ imagePrompt }
				generateImage={ generateImage }
				processing={ processing }
				setImagePrompt={ setImagePrompt }
			/>;
		}

		return <SEOPanel
			keyword={ keyword }
			processAnalyze={ processAnalyze }
			processing={ processing }
			seoData={ seoData }
			setKeyword={ setKeyword }
		/>;
	};

	/**
	 * On tab select, if showing the SEO tab, make sure we force update the SEO data, so the charts are updated.
	 *
	 * @param {string} tabName
	 */
	const onTabSelect = ( tabName ) => {
		if ( 'seo' === tabName ) {
			setSeoData( { ...seoData } );
		}
	};

	return (
		<PluginSidebar name="fuzion-sidebar" title={ __( 'Fuzion AI', 'fuzion' ) }>
			<TabPanel
				activeClass="is-active"
				tabs={ tabs }
				onSelect={ onTabSelect }
			>
				{ ( tab ) => renderTab( tab ) }
			</TabPanel>
			{ showModal && <AiContentModal
				cancelCb={ cancelGenerate }
				content={ response }
				processCb={ processGenerate }
				processing={ processing }
				selectCb={ selectItem }
				selected={ checked }
				submitCb={ addToEditor }
			/> }
		</PluginSidebar>
	);
};

export default AIPluginSidebar;
