/**
 * Product page metabox scripts
 *
 * @since 1.3.0
 */

// Internal dependencies
import './../scss/components/product-description.scss';
import API from './modules/api';
import Notice from './modules/notice';
import { getString } from './modules/utils';

( function( $ ) {
	$( function() {
		const notice = new Notice();
		const api = new API();

		$( '#fuzion_ai' )
			.on( 'change input', 'input', function() {
				const rangeValue = parseInt( $( this ).val() );
				const sibling = $( this ).siblings( 'input' );

				if ( rangeValue > 0 ) {
					sibling.val( rangeValue );

					if ( sibling.attr( 'type' ) === 'range' ) {
						sibling.trigger( 'change' );
						return;
					}
				}

				if ( $( this ).attr( 'type' ) === 'range' ) {
					const min = parseInt( $( this ).attr( 'min' ) ) || 1;
					const max = parseInt( $( this ).attr( 'max' ) ) || 100;
					const backgroundSize = ( ( rangeValue - min ) * 100 / ( max - min ) ) + '% 100%';
					$( this ).css( 'background-size', backgroundSize );
				}
			} )
			.on( 'click', '#generate-content', function() {
				toggleLoading( true );
				toggleButtonState( true );

				const productTitle = $( '#title' ).val();
				if ( ! productTitle ) {
					notice.error( getString( 'titleIsRequired' ), false );
					toggleLoading( false );
					toggleButtonState( false );
					return;
				}

				const attributes = $( '#fuzion_ai .attributes-group' ).find( 'input[type="checkbox"]:checked' ).map( ( i, el ) => el.name ).get();

				api.post(
					'ai_product_description',
					{
						product_id: $( '#post_ID' ).val(),
						attributes
					}
				).then( ( response ) => {
					if ( ! api.processResponse( response ) ) {
						return;
					}
					setProductDescription( response.data.message[ 0 ] );
				} ).finally( () => {
					toggleLoading( false );
					toggleButtonState( false );
				} );
			} )
			.find( 'input[type="range"]' ).trigger( 'change' );

		/**
		 * Set product description to editor
		 *
		 * @param {string} content
		 * @since 1.3.0
		 */
		function setProductDescription( content ) {
			const editorId = 'content';

			if ( $( `#wp-${ editorId }-wrap` ).hasClass( 'tmce-active' ) ) {
				window.tinymce.get( editorId ).setContent( content );
			} else {
				$( `#${ editorId }` ).val( content );
			}
		}

		/**
		 * Hide / Show spinner on submit
		 *
		 * @param {boolean} visible
		 * @since 1.3.0
		 */
		function toggleLoading( visible ) {
			$( '#fuzion_ai' ).find( '.spinner' ).toggleClass( 'is-active', !! visible );
		}

		/**
		 * Disable / Enable button on submit
		 *
		 * @param {boolean} disabled
		 * @since 1.3.0
		 */
		function toggleButtonState( disabled ) {
			$( '#generate-content' ).attr( 'disabled', !! disabled );
		}
	} );
}( window.jQuery ) );
