/**
 * Functions for notice management.
 *
 * @since 1.0.0
 */

'use strict';

export default class Notice {
	/**
	 * Try to add errors to forms, based on fieds.
	 *
	 * @since 1.0.0
	 * @param {Object} errors Object of errors, where key - form element ID, value - error text.
	 */
	showFormErrors( errors = {} ) {
		jQuery( '.ai-ajax-error' ).remove();

		for ( const [ type, error ] of Object.entries( errors ) ) {
			const el = jQuery( `input#${ type }` );

			if ( ! el ) {
				continue;
			}

			el.attr( 'aria-invalid', true );
			el.after( `<small class="form-error ai-ajax-error">${ error }</small>` );
		}
	}

	/**
	 * Show error notice.
	 *
	 * @since 1.0.0
	 * @param {string}  message  Message text.
	 * @param {boolean} autoHide Auto hide message.
	 */
	error( message = '', autoHide = false ) {
		this.#show( message, 'error', autoHide );
	}

	/**
	 * Show success notice.
	 *
	 * @since 1.0.0
	 * @param {string}  message  Message text.
	 * @param {boolean} autoHide Auto hide message.
	 */
	success( message = '', autoHide = false ) {
		this.#show( message, 'success', autoHide );
	}

	/**
	 * Internal show message method.
	 *
	 * @since 1.0.0
	 * @param {string}  message  Message text.
	 * @param {string}  type     Message type.
	 * @param {boolean} autoHide Auto hide message.
	 */
	#show( message, type = 'success', autoHide = false ) {
		const notice = jQuery( '#ai-ajax-notice' );

		notice.empty();

		notice.append( `<p>${ message }</p>` );
		notice.removeClass( 'notice-success notice-error' )
			.addClass( 'notice-' + type );

		notice.show();
		if ( autoHide ) {
			setTimeout( () => notice.slideUp( 'slow' ), 5000 );
		}
	}
}
