/**
 * Functions for API calls.
 *
 * @since 1.0.0
 */

'use strict';

import Notice from './notice';

/* global Fuzion */
/* global ajaxurl */

export default class API {
	/**
	 * Do AJAX request to WordPress.
	 *
	 * @since 1.0.0
	 * @param {string} action Registered AJAX action.
	 * @param {Object} data   Additional data that needs to be passed in POST request.
	 * @return {Promise<unknown>} Return data.
	 */
	post( action, data = {} ) {
		return new Promise( ( resolve, reject ) => {
			jQuery.ajax( {
				url: ajaxurl,
				type: 'POST',
				data: { _ajax_nonce: Fuzion.nonce, action, ...data },
				success( response ) {
					resolve( response );
				},
				error( error ) {
					reject( error );
				},
			} );
		} );
	}

	/**
	 * Process API response.
	 *
	 * @since 1.0.0
	 *
	 * @param {Object} response API response.
	 * @return {boolean} Return false on issues with response. True on success.
	 */
	processResponse( response ) {
		const notice = new Notice();

		// Unsuccessful request.
		if ( ! response.success ) {
			notice.error( response.data );
			return false;
		}

		// Exception.
		if ( response.data.exception && response.data.message ) {
			notice.error( response.data.message );
			return false;
		}

		// Error in the form.
		if ( response.data.errors && response.data.message ) {
			notice.error( response.data.message );
			notice.showFormErrors( response.data.errors );
			return false;
		}

		return true;
	}
}
