/**
 * WordPress dependencies
 */
import { Button, Icon, Panel, PanelBody, PanelRow, TextControl, Tooltip } from '@wordpress/components';
import { __ } from '@wordpress/i18n';

/**
 * External dependencies
 */
import classNames from 'classnames';

/**
 * SEO analysis panel.
 *
 * @param {string}   keyword                       Keyword to analyze.
 * @param {Function} processAnalyze                Analyze keyword.
 * @param {boolean}  processing                    Processing state.
 * @param {Object}   seoData                       SEO data for selected keyword.
 * @param {number}   seoData.clickedAnyResult
 * @param {number}   seoData.percentDesktopSearches
 * @param {number}   seoData.percentMobileSearches
 * @param {number}   seoData.percentPaidClicks
 * @param {number}   seoData.percentPaidClicksChartValue
 * @param {number}   seoData.rankingDifficulty
 * @param {number}   seoData.searchVolume
 * @param {number}   seoData.totalMonthlyClicks
 * @param {Function} setKeyword                    Set keyword.
 *
 * @return {JSX.Element} SEO analysis panel.
 */
const SEOPanel = ( { keyword, processAnalyze, processing, seoData, setKeyword } ) => {
	/**
	 * Has seo data saved in component state.
	 *
	 * @return {boolean} True if there is data, false otherwise.
	 */
	const hasSeoData = () => {
		return Object.entries( seoData ).length > 0;
	};

	/**
	 * Get Mobile vs Desktop status label.
	 *
	 * @return {string} Status label.
	 */
	const getMobileVsDesktopLabel = () => {
		if ( 'undefined' === typeof seoData.percentMobileSearches || seoData.percentMobileSearches <= 0.2 ) {
			return __( 'Not Enough Data', 'fuzion' );
		} else if ( seoData.percentMobileSearches <= 0.4 ) {
			return __( 'Low Mobile', 'fuzion' );
		} else if ( seoData.percentMobileSearches <= 0.6 ) {
			return __( 'Normal', 'fuzion' );
		} else if ( seoData.percentMobileSearches <= 0.75 ) {
			return __( 'High Mobile', 'fuzion' );
		}

		return __( 'Very High Mobile', 'fuzion' );
	};

	return <Panel>
		<PanelBody title={ __( 'SEO Analysis', 'fuzion' ) }>
			<TextControl
				help={ __( 'Enter a keyword to analyze', 'fuzion' ) }
				label={ __( 'Keyword', 'fuzion' ) }
				onChange={ ( value ) => setKeyword( value ) }
				value={ keyword }
			/>
			<Button
				disabled={ processing || ! keyword }
				isBusy={ processing }
				onClick={ processAnalyze }
				text={ processing ? __( 'Analyzing...', 'fuzion' ) : __( 'Analyze keyword', 'fuzion' ) }
				variant="secondary"
			/>
		</PanelBody>
		<PanelBody title={ __( 'Results', 'fuzion' ) } className={ classNames( 'fuzion-seo-charts', { hidden: ! hasSeoData() } ) }>
			<PanelRow>
				<span className="fuzion-tooltip">
					{ __( 'Monthly volume', 'fuzion' ) }
					<Tooltip position="top center" delay="0" text={ __( 'Number of searches done past month across Google (based on the selected region). Data is combined with other sources, so might not always be identical to Google\'s metrics for search volume.', 'fuzion' ) }>
						<span><Icon icon="info-outline" /></span>
					</Tooltip>
				</span>
				<strong>{ seoData.searchVolume }</strong>
			</PanelRow>
			<canvas id="monthlyVolume" width="100vw" height="25vw"></canvas>
			<PanelRow>
				<span className="fuzion-tooltip">
					{ __( 'Estimated clicks', 'fuzion' ) }
					<Tooltip position="top center" delay="0" text={ __( 'Total number of all clicks (organic and paid) made on the SERP over the past month.', 'fuzion' ) }>
						<span><Icon icon="info-outline" /></span>
					</Tooltip>
				</span>
				<strong>{ seoData.totalMonthlyClicks }</strong>
			</PanelRow>
			<PanelRow>
				<span className="fuzion-tooltip">
					{ __( 'Mobile vs Desktop', 'fuzion' ) }
					<Tooltip position="top center" delay="0" text={ __( 'This chart reflects a percentage of searches. There are some keywords that get searched from mobile devices more often than others. And, some searches are more likely to be done from a desktop. A strong swing in either direction might influence the kind of content you create or the format you use. If this keyword\'s mobile searches stray far from usual patterns, you will see low bars on the indicator below.', 'fuzion' ) }>
						<span><Icon icon="info-outline" /></span>
					</Tooltip>
				</span>
				<canvas id="mobileVsDesktop" width="140" height="45"></canvas>
			</PanelRow>
			<PanelRow>
				<small>{ getMobileVsDesktopLabel() }</small>
				<svg height="20" width="29">
					<rect className={ classNames( { 'chart-grey': seoData.percentMobileSearches <= 0.2, 'chart-indigo': seoData.percentMobileSearches > 0.2 } ) } height="5" width="5" x="0" y="15" rx="1"></rect>
					<rect className={ classNames( { 'chart-grey': seoData.percentMobileSearches <= 0.4, 'chart-indigo': seoData.percentMobileSearches > 0.4 } ) } height="10" width="5" x="8" y="10" rx="1"></rect>
					<rect className={ classNames( { 'chart-grey': seoData.percentMobileSearches <= 0.6, 'chart-indigo': seoData.percentMobileSearches > 0.6 } ) } height="15" width="5" x="16" y="5" rx="1"></rect>
					<rect className={ classNames( { 'chart-grey': seoData.percentMobileSearches <= 0.75, 'chart-indigo': seoData.percentMobileSearches > 0.75 } ) } height="20" width="5" x="24" y="0" rx="1"></rect>
				</svg>
			</PanelRow>
			<br />
			<span className="fuzion-tooltip">
				{ __( 'Clicked any result', 'fuzion' ) }
				<Tooltip position="top center" delay="0" text={ __( 'Some SERPs return enough information that the user does not have to click any results. Or, the keyword might be broad or ambiguous, leaving the user to abandon the SERP when they don\'t see what they hoped to find. In that case, this metric will appear as a lower percentage.', 'fuzion' ) }>
					<span><Icon icon="info-outline" /></span>
				</Tooltip>
			</span>
			<canvas id="clickedAnyResult" width="100vw" height="25vw"></canvas>
			<div className="chart-clicked-any-results">
				<div><div></div><div>{ __( 'Low', 'fuzion' ) }</div></div>
				<div>{ Math.round( seoData.clickedAnyResult * 100 ) }%</div>
				<div><div></div><div>{ __( 'High', 'fuzion' ) }</div></div>
			</div>
			<br />
			<span className="fuzion-tooltip">
				{ __( 'Paid clicks', 'fuzion' ) }
				<Tooltip position="top center" delay="0" text={ __( 'A search for "flower delivery" will probably get more clicks to the ads than a search for "capital of Bolivia" would. Some searches are more transactional vs informational, and this chart helps you to see where this keyword falls on the scale. It\'s rare for ads to get more than 30% of total clicks, so that would rate as a "high" paid click rate.', 'fuzion' ) }>
					<span><Icon icon="info-outline" /></span>
				</Tooltip>
			</span>
			<canvas id="paidClicks" width="100vw" height="25vw"></canvas>
			<div className="chart-paid-clicks">
				<div><div></div><div>{ __( 'Low', 'fuzion' ) }<br />0‑3%</div></div>
				<div>{ Math.round( seoData.percentPaidClicks * 100 ) }%</div>
				<div><div></div><div>{ __( 'High', 'fuzion' ) }<br />13%+</div></div>
			</div>
			<br />
			<span className="fuzion-tooltip">
				{ __( 'Difficulty', 'fuzion' ) }
				<Tooltip position="top center" delay="0" text={ __( 'How difficult it would be to rank on this keyword. The score is based on a scale of 0-100 (with 100 being the most difficult to rank for). Compare this number to other keywords you\'re targeting to get an idea of how to prioritize your SEO campaign. Keyword difficulty takes into account the strength of the domains, on-page signals like "keyword in title", and the number of .gov and .edu domains.', 'fuzion' ) }>
					<span><Icon icon="info-outline" /></span>
				</Tooltip>
			</span>
			<canvas id="difficulty" width="100vw" height="25vw"></canvas>
			<div className="chart-difficulty">
				<div id="rankingDifficultySpan">
					<span>{ seoData.rankingDifficulty }</span>
				</div>
			</div>
		</PanelBody>
	</Panel>;
};

export default SEOPanel;
