/**
 * WordPress dependencies
 */
import { Button, Panel, PanelBody } from '@wordpress/components';
import { Fragment, useState, useEffect } from '@wordpress/element';
import { CSSTransition, TransitionGroup } from 'react-transition-group';
import { addAction, removeAction } from '@wordpress/hooks';

/**
 * Internal dependencies
 */
import Prompt from './slides/prompt';
import ContentTypes from './slides/content-type';

/**
 * Settings panel.
 *
 * @param {Object} props
 * @return {JSX.Element} Settings panel.
 */
const GeneralPanel = ( props ) => {
	const {
		processing,
	} = props;

	const slides = [ ContentTypes, Prompt ];
	const [ currentSlide, setSlide ] = useState( 0 );
	const [ slideDirection, setSlideDirection ] = useState( '' );

	const nextSlide = () => {
		const next = currentSlide === slides.length - 1 ? 0 : currentSlide + 1;
		setSlide( next );
		setSlideDirection( 'right' );
	};

	const prevSlide = () => {
		setSlide( currentSlide - 1 );
		setSlideDirection( 'left' );
	};

	const renderCurrentSlide = () => {
		const Slide = slides[ currentSlide ];
		return <Slide { ...{ ...props, nextSlide, prevSlide } } />;
	};

	useEffect( () => {
		addAction( 'aiContent.complete', 'fuzion/reset-slide', nextSlide );

		return () => {
			removeAction( 'aiContent.complete', 'fuzion/reset-slide' );
		};
	} );

	return <Panel>
		<PanelBody>
			<div style={ {
				width: '100%',
				display: 'flex',
				background: 'white',
				flexDirection: 'row',
				flexWrap: 'nowrap'
			} }
			>
				<TransitionGroup className="slide-group" style={ { width: '100%' } }>
					<CSSTransition
						classNames="slide"
						timeout={ { enter: 350, exit: 350 } }
						key={ currentSlide }
					>
						<div className={ slideDirection }>
							{ renderCurrentSlide() }
						</div>
					</CSSTransition>
				</TransitionGroup>
			</div>
			{ ! processing && currentSlide >= 1 &&
				<Fragment>
					<Button icon="arrow-left-alt" disabled={ processing || currentSlide < 1 } onClick={ prevSlide } />
					<Button icon="arrow-right-alt" disabled={ processing || currentSlide <= ( slides.length - 1 ) } onClick={ nextSlide } />
				</Fragment>
			}
		</PanelBody>
	</Panel>;
};

export default GeneralPanel;
