/**
 * WordPress dependencies
 */
import { __ } from '@wordpress/i18n';
import { Button, CheckboxControl, Modal } from '@wordpress/components';

/**
 * AI content selection modal.
 *
 * @since 1.1.0
 *
 * @param {Object}   props
 * @param {Function} props.cancelCb   Callback to run when cancel button is clicked.
 * @param {Array}    props.content    Array of content to display.
 * @param {Function} props.processCb  Callback to run when "Generate again" button is clicked.
 * @param {boolean}  props.processing Whether or not the modal is processing.
 * @param {Function} props.selectCb   Callback to run when content is selected.
 * @param {Array}    props.selected   Array of selected content.
 * @param {Function} props.submitCb   Callback to run when "Add to editor" button is clicked.
 *
 * @return {JSX.Element} Modal.
 */
const AiContentModal = ( { cancelCb, content, processCb, processing, selectCb, selected, submitCb } ) => {
	return <Modal
		className="fuzion-editor-modal"
		onRequestClose={ cancelCb }
		title={ __( 'Generated content', 'fuzion' ) }
	>
		{ content.length === 1 && <p>{ content[ 0 ] }</p> }
		{ content.length > 1 && content.map( ( el, index ) => {
			return <CheckboxControl
				key={ index }
				label={ el }
				checked={ selected.includes( index ) }
				onChange={ () => selectCb( index ) }
			/>;
		} ) }
		<Button
			disabled={ processing }
			icon="update"
			isBusy={ processing }
			onClick={ processCb }
			variant="secondary"
		>
			{ __( 'Generate again', 'fuzion' ) }
		</Button>
		<Button
			disabled={ processing }
			icon="editor-paste-text"
			isBusy={ processing }
			onClick={ submitCb }
			variant="primary"
		>
			{ __( 'Add to editor', 'fuzion' ) }
		</Button>
	</Modal>;
};

export default AiContentModal;
