/**
 * WordPress dependencies
 */
import { __ } from '@wordpress/i18n';
import { Button, Modal, RadioControl, RangeControl, TextareaControl } from '@wordpress/components';

/**
 * Code edit options modal.
 *
 * @since 1.4.0
 *
 * @param {Object}   props
 * @param {Function} props.cancelCb        Callback to run when cancel button is clicked.
 * @param {number}   props.credits         Credits to use.
 * @param {Function} props.generate        Generate callback.
 * @param {string}   props.instructions    Prompt value.
 * @param {boolean}  props.processing      Whether or not the code generations is in progress.
 * @param {boolean}  props.replace         Replace value.
 * @param {Function} props.setCredits      Callback to set credits.
 * @param {Function} props.setInstructions Callback to update the prompt value.
 * @param {Function} props.setReplace      Callback to update the replacement status value.
 *
 * @return {JSX.Element} Modal.
 */
const CodeEditOptions = ( { cancelCb, credits, generate, instructions, processing, replace, setCredits, setInstructions, setReplace } ) => {
	return <Modal
		className="fuzion-editor-modal"
		onRequestClose={ cancelCb }
		title={ __( 'Code AI options', 'fuzion' ) }
	>
		<TextareaControl
			help={ __( 'Explain what should be done with your code, for example: "Add a docstring" or "Fill in the function body".', 'fuzion' ) }
			label={ __( 'Prompt', 'fuzion' ) }
			onChange={ ( value ) => setInstructions( value ) }
			value={ instructions }
		/>

		<RangeControl
			help={ __( 'You can think of credits as pieces of words, where 1,000 credits is about 750 words. It is recommended to set this value to at least 300. If the set value is less than the required credits, resulting code will be truncated. This paragraph is around 61 credits.', 'fuzion' ) }
			label={ __( 'Maximum credits to use', 'fuzion' ) }
			max={ 2048 }
			min={ 1 }
			onChange={ ( value ) => setCredits( value ) }
			value={ credits }
		/>

		<RadioControl
			help={ __( 'Select where the generate code should be placed. Replace - will overwrite the current core/code block; new - will add the results to the blocks under the core/code block.', 'fuzion' ) }
			label={ __( 'Results placement', 'fuzion' ) }
			onChange={ ( value ) => setReplace( value === 'true' ) }
			options={ [
				{ label: __( 'Replace', 'fuzion' ), value: true },
				{ label: __( 'New blocks', 'fuzion' ), value: false },
			] }
			selected={ replace }
		/>

		<Button
			disabled={ processing }
			isBusy={ processing }
			onClick={ generate }
			variant="primary"
		>
			{ __( 'Generate', 'fuzion' ) }
		</Button>
	</Modal>;
};

export default CodeEditOptions;
