/**
 * Internal dependencies
 */
import Chart from 'chart.js/auto';

/**
 * Monthly volume chart.
 */
export default class monthlyVolumeChart {
	constructor( value ) {
		this.bars = [ 10, 10.3, 10.6, 11, 11.3, 11.6, 12, 12.6, 13, 13.5, 14.1, 14.7, 15.3, 16, 16.6, 17.2, 18, 18.9, 19.6, 20.4, 21.4, 22.4, 23.5, 24.5, 25.7, 27, 28.2, 29.7, 31.1, 32.6, 34.3, 36.1, 38.1, 40.2, 42.4, 44.9, 47.4, 50.5, 53.5, 58 ];
		this.value = this.calculateValue( value );
		this.renderChart();
	}

	/**
	 * Calculate the percentage equivalent of the volume value.
	 * For example, a monthly volume of 360 would be 0.2777777777777778 (equal to 27.77777777777778%).
	 * Later on we will use this computed value to calculate how many bars are highlighted.
	 *
	 * @param {number} value Current monthly volume value.
	 * @return {number} Percentage equivalent of the volume value.
	 */
	calculateValue( value ) {
		const e = Math.max( 20, value );

		if ( ! Number.isFinite( e ) ) {
			return 0;
		}

		const s = [ 10, 20, 50, 100, 200, 500, 1e3, 2e3, 5e3, 1e4, 5e4, 1e5, 2e5, 5e5, 1e6, 2e6, 5e6, 1e7 ];
		for ( let val = 0, a = s.length; val < a; val += 1 ) {
			const r = s[ val + 1 ] || 1 / 0;
			if ( e >= ( s[ val - 1 ] || 0 ) && e < r ) {
				return ( val + 1 ) / a;
			}
		}

		return 1;
	}

	/**
	 * Calculate the number of bars to highlight.
	 *
	 * @param {number} index Current bar index.
	 * @return {boolean} Highlight status.
	 */
	isHighlighted = ( index ) => {
		return ( this.bars.length - 1 ) * this.value >= index;
	};

	/**
	 * Render the chart.
	 */
	renderChart() {
		const chartExists = Chart.getChart( 'monthlyVolume' );
		if ( chartExists ) {
			chartExists.destroy();
		}

		const chart = document.getElementById( 'monthlyVolume' ).getContext( '2d' );
		new Chart( chart, {
			type: 'bar',
			data: {
				labels: this.bars,
				datasets: [ {
					backgroundColor: ( el ) => {
						return this.isHighlighted( el.dataIndex ) ? '#6366f1' : '#d1d5db';
					},
					data: this.bars,
				} ]
			},
			options: {
				barThickness: 4,
				plugins: {
					legend: {
						display: false
					},
					tooltip: {
						enabled: false
					}
				},
				responsive: true,
				scales: {
					x: {
						display: false,
					},
					y: {
						display: false,
					}
				},
			},
		} );
	}
}
