'use strict';

/**
 * Internal dependencies
 */
import '../scss/admin.scss';
import API from './modules/api';
import Notice from './modules/notice';
import { getLink, getString } from './modules/utils';

/**
 * Fuzion class.
 *
 * @since 1.0.0
 */
export default class Fuzion {
	/**
	 * Class constructor.
	 */
	constructor() {
		this.api = new API();
		this.notice = new Notice();

		this.mapInputs();
		this.processRegistrationForm();
		this.processLoginForm();
		this.processKeyUpdate();
		this.handleStatsSync();
		this.handleLogout();
	}

	/**
	 * Hide errors when we type in password inputs.
	 *
	 * @since 1.0.0
	 */
	mapInputs() {
		jQuery( 'input[type="text"], input[type="email"], input[type="password"]' ).on( 'input', function() {
			const inputType = jQuery( this ).attr( 'type' );
			const inputs = jQuery( `input[type="${ inputType }"]` );

			inputs.removeAttr( 'aria-invalid' );
			jQuery( '.ai-ajax-error' ).remove();

			// If passwords match - show tick icon.
			if ( 'password' === inputType && jQuery( '#password' ).val() === jQuery( '#password_confirmation' ).val() ) {
				inputs.attr( 'aria-invalid', false );
			}
		} );
	}

	/**
	 * Process registration form.
	 *
	 * @since 1.0.0
	 */
	processRegistrationForm() {
		jQuery( '#ai-register-form' ).on( 'submit', ( e ) => {
			e.preventDefault();
			this.removeFormErrors();

			const submitButton = jQuery( '#ai-register-btn' );
			submitButton.attr( 'aria-busy', true );

			const password = jQuery( '#password' );
			const passwordConfirmation = jQuery( '#password_confirmation' );

			// Check that passwords match.
			if ( password.val() !== passwordConfirmation.val() ) {
				this.notice.showFormErrors( { password_confirmation: getString( 'passwordsNotMatch' ) } );
				jQuery( 'input[type="password"]' ).attr( 'aria-invalid', true );
				submitButton.attr( 'aria-busy', false );
				return;
			}

			const data = {
				name: jQuery( '#name' ).val(),
				email: jQuery( '#email' ).val(),
				password: password.val(),
				password_confirmation: passwordConfirmation.val(),
			};

			this.api.post( 'ai_register', data )
				.then( ( response ) => {
					if ( ! this.api.processResponse( response ) ) {
						return;
					}

					window.location.href = getLink( 'pluginURL' );
				} )
				.finally( () => {
					submitButton.attr( 'aria-busy', false );
				} )
				.catch( window.console.log );
		} );
	}

	/**
	 * Process API key update form..
	 *
	 * @since 1.0.0
	 */
	processKeyUpdate() {
		jQuery( '#ai-api-key-form' ).on( 'submit', ( e ) => {
			e.preventDefault();

			const submitButton = jQuery( '#ai-update-api-key-btn' );
			submitButton.attr( 'aria-busy', true );

			const data = {
				key: jQuery( '#api-key' ).val(),
			};

			this.api.post( 'ai_update_key', data )
				.then( ( response ) => {
					if ( ! this.api.processResponse( response ) ) {
						return;
					}

					window.location.href = getLink( 'pluginURL' );
				} )
				.finally( () => submitButton.attr( 'aria-busy', false ) )
				.catch( window.console.log );
		} );
	}

	/**
	 * Process login form.
	 *
	 * @since 1.0.0
	 */
	processLoginForm() {
		jQuery( '#ai-login-form' ).on( 'submit', ( e ) => {
			e.preventDefault();
			this.removeFormErrors();

			const submitButton = jQuery( '#ai-login-btn' );
			submitButton.attr( 'aria-busy', true );

			const data = {
				email: jQuery( '#email' ).val(),
				password: jQuery( '#password' ).val(),
			};

			this.api.post( 'ai_login', data )
				.then( ( response ) => {
					if ( ! this.api.processResponse( response ) ) {
						return;
					}

					window.location.href = getLink( 'pluginURL' );
				} )
				.finally( () => {
					submitButton.attr( 'aria-busy', false );
				} )
				.catch( window.console.log );
		} );
	}

	/**
	 * Handle stats sync.
	 *
	 * @since 1.0.0
	 */
	handleStatsSync() {
		jQuery( '#ai-ajax-sync-stats' ).on( 'click', ( e ) => {
			e.preventDefault();

			const submitButton = jQuery( '#ai-ajax-sync-stats' );
			submitButton.attr( 'aria-busy', true );

			this.api.post( 'ai_sync_stats' )
				.then( ( response ) => {
					if ( ! this.api.processResponse( response ) ) {
						return;
					}

					this.notice.success( getString( 'statsSynced' ), true );

					for ( const key in response.data ) {
						jQuery( '#' + key ).text( response.data[ key ] );
					}
				} )
				.finally( () => {
					submitButton.attr( 'aria-busy', false );
				} )
				.catch( window.console.log );
		} );
	}

	/**
	 * Handle "Log out" button click.
	 *
	 * @since 1.0.0
	 */
	handleLogout() {
		jQuery( '#ai-ajax-logout' ).on( 'click', ( e ) => {
			e.preventDefault();

			jQuery( '#ai-ajax-logout' ).attr( 'aria-busy', true );

			this.api.post( 'ai_logout' )
				.then( () => {
					window.location.href = getLink( 'pluginURL' );
				} )
				.catch( window.console.log );
		} );
	}

	/**
	 * Remove form errors and hide notices.
	 *
	 * @since 1.0.0
	 */
	removeFormErrors() {
		jQuery( '.ai-ajax-error' ).remove();
		jQuery( '#ai-ajax-notice' ).hide();
	}
}

const FuzionAI = new Fuzion();
window.FuzionAI = FuzionAI;
