<?php
/**
 * The class that defines the User model
 *
 * Manage user data, API tokens.
 *
 * @link https://getfuzion.io
 *
 * @package Fuzion
 * @subpackage Fuzion/App/Models
 * @author Fuzion AI
 * @since 1.0.0
 */

namespace Fuzion\App\Models;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * User class.
 *
 * @since 1.0.0
 */
class User {

	/**
	 * Default token data.
	 *
	 * @since 1.0.0
	 * @access private
	 * @var array|int[]
	 */
	private static $data = array(
		'registered' => false,
		'verified'   => false,
	);

	/**
	 * Set API token value.
	 *
	 * @since 1.0.0
	 *
	 * @param string $value  API token.
	 *
	 * @return void
	 */
	public static function set_token( string $value ) {

		update_option( 'fuzion-api-token', $value, false );

		// If we have the token - it means user is registered and verified.
		self::set_verified();

	}

	/**
	 * Get API token value.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public static function get_token(): string {

		$token = get_option( 'fuzion-api-token', '' );

		if ( defined( 'FUZION_API_TOKEN' ) && FUZION_API_TOKEN ) {
			$token = FUZION_API_TOKEN;
		}

		return $token;

	}

	/**
	 * Check if API token is present.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public static function has_api_token(): bool {
		return ! empty( self::get_token() );
	}

	/**
	 * Mark user as registered.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function set_registered() {

		$options = get_option( 'fuzion-registered', self::$data );

		$options['registered'] = true;

		update_option( 'fuzion-registered', $options, false );

	}

	/**
	 * Mark user as email-verified.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function set_verified() {

		$options = get_option( 'fuzion-registered', self::$data );

		// If, for some reason, user is not marked as registered - fix it.
		if ( ! isset( $options['registered'] ) || ! $options['registered'] ) {
			$options['registered'] = true;
		}

		$options['verified'] = true;

		update_option( 'fuzion-registered', $options, false );

	}

	/**
	 * Check if user is registered.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public static function is_registered(): bool {

		$status = get_option( 'fuzion-registered' );

		if ( ! $status ) {
			return false;
		}

		return isset( $status['registered'] ) && $status['registered'];

	}

	/**
	 * Check if user email is verified.
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public static function is_email_verified(): bool {

		$status = get_option( 'fuzion-registered' );

		if ( ! $status ) {
			return false;
		}

		return isset( $status['verified'] ) && $status['verified'];

	}

}
