<?php
/**
 * The class that defines the Credit model
 *
 * Get and set available credits.
 *
 * @link https://getfuzion.io
 *
 * @package Fuzion
 * @subpackage Fuzion/App/Models
 * @author Fuzion AI
 * @since 1.0.0
 */

namespace Fuzion\App\Models;

use stdClass;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Credit class.
 *
 * @since 1.0.0
 */
class Credit {

	/**
	 * Default credits.
	 *
	 * @since 1.0.0
	 * @access private
	 * @var array|int[]
	 */
	private static $data = array(
		'content' => array(
			'monthly'   => 2000,
			'remaining' => 2000,
			'requests'  => 0,
		),
		'images'  => array(
			'monthly'   => 5,
			'remaining' => 5,
		),
		'plan'    => 'Free',
		'seo'     => array(
			'monthly'   => 5,
			'remaining' => 5,
		),
	);

	/**
	 * Get monthly credits.
	 *
	 * @since 1.0.0
	 *
	 * @param string $type  Credit type.
	 *
	 * @return int
	 */
	public static function get_monthly_credits( string $type = 'content' ): int {

		$credits = get_option( 'fuzion-credits', self::$data );

		if ( ! isset( $credits[ $type ] ) ) {
			return 0;
		}

		return $credits[ $type ]['monthly'] ?? 0;

	}

	/**
	 * Get plan name.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public static function get_plan_name(): string {

		$credits = get_option( 'fuzion-credits', self::$data );
		return $credits['plan'] ?? __( 'Undefined', 'fuzion' );

	}

	/**
	 * Get total requests count.
	 *
	 * @since 1.0.0
	 *
	 * @param string $type  Credit type.
	 *
	 * @return int
	 */
	public static function get_requests_count( string $type = 'content' ): int {

		$credits = get_option( 'fuzion-credits', self::$data );

		if ( ! isset( $credits[ $type ] ) ) {
			return 0;
		}

		return $credits[ $type ]['requests'] ?? 0;

	}

	/**
	 * Get available credits.
	 *
	 * @since 1.0.0
	 *
	 * @param string $type  Credits type.
	 *
	 * @return int
	 */
	public static function get_available_credits( string $type = 'content' ): int {

		$credits = get_option( 'fuzion-credits', self::$data );

		if ( ! isset( $credits[ $type ] ) ) {
			return 0;
		}

		return $credits[ $type ]['remaining'] ?? 0;

	}

	/**
	 * Set available credits.
	 *
	 * @since 1.0.0
	 *
	 * @param int    $credits  Credits amount.
	 * @param string $type     Credits type.
	 *
	 * @return void
	 */
	public static function set_available_credits( int $credits, string $type = 'content' ) {

		$available_credits = get_option( 'fuzion-credits', self::$data );

		if ( ! isset( $available_credits[ $type ] ) ) {
			return;
		}

		$available_credits[ $type ]['remaining'] = $credits;

		update_option( 'fuzion-credits', $available_credits, false );

	}

	/**
	 * Update stats.
	 *
	 * @since 1.0.0
	 *
	 * @param stdClass $stats  Stats object from API.
	 *
	 * @return void
	 */
	public static function update_stats( stdClass $stats ) {

		$credits = get_option( 'fuzion-credits', self::$data );

		// Update limits.
		if ( isset( $stats->limits ) ) {
			foreach ( $stats->limits as $type => $limit ) {
				$credits[ $type ]['monthly'] = $limit;
			}
		}

		// Update remaining.
		if ( isset( $stats->credits ) ) {
			foreach ( $stats->credits as $type => $remaining ) {
				$credits[ $type ]['remaining'] = $remaining;
			}
		}

		// Update content requests count.
		if ( isset( $stats->requests ) ) {
			$credits['content']['requests'] = $stats->requests;
		}

		update_option( 'fuzion-credits', $credits, false );

	}

}
