<?php
/**
 * The file that defines the core plugin class
 *
 * Core plugin class that is responsible with initializing all the plugin functionality and loading all required modules.
 *
 * @link https://getfuzion.io
 *
 * @package Fuzion
 * @subpackage Fuzion/App
 * @author Fuzion AI
 * @since 1.0.0
 */

namespace Fuzion\App;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The core plugin class.
 *
 * @since 1.0.0
 */
class Core {

	/**
	 * Editor instance.
	 *
	 * @since 1.0.0
	 * @var Editor $editor
	 */
	protected $editor;

	/**
	 * Ajax instance.
	 *
	 * @since 1.0.0
	 * @var Ajax $ajax
	 */
	protected $ajax;

	/**
	 * Admin instance.
	 *
	 * @since 1.0.0
	 * @var Admin\Admin $admin
	 */
	protected $admin;

	/**
	 * Plugin instance.
	 *
	 * @since 1.0.0
	 * @access private
	 * @var null|Core $instance  Plugin instance.
	 */
	private static $instance = null;

	/**
	 * Get plugin instance.
	 *
	 * @since 1.0.0
	 *
	 * @return Core
	 */
	public static function get_instance(): Core {

		if ( ! self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;

	}

	/**
	 * Define the core functionality of the plugin.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {

		$this->load_libs();

		$this->editor = new Editor();
		$this->ajax   = new Ajax( new Api\Auth() );
		$this->admin  = new Admin\Admin();

	}

	/**
	 * Load all required files.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	private function load_libs() {

		// Traits and editor.
		require_once __DIR__ . '/traits/trait-utils.php';
		require_once __DIR__ . '/class-editor.php';

		// Models.
		require_once __DIR__ . '/models/class-credit.php';
		require_once __DIR__ . '/models/class-user.php';
		require_once __DIR__ . '/models/class-seo.php';
		require_once __DIR__ . '/models/class-image.php';
		require_once __DIR__ . '/models/class-product.php';

		// API.
		require_once __DIR__ . '/api/class-request.php';
		require_once __DIR__ . '/api/class-auth.php';
		require_once __DIR__ . '/api/class-content.php';
		require_once __DIR__ . '/api/class-seo.php';
		require_once __DIR__ . '/api/woocommerce/class-product.php';

		// Ajax and admin.
		require_once __DIR__ . '/class-ajax.php';
		require_once __DIR__ . '/admin/class-view.php';
		require_once __DIR__ . '/admin/class-admin.php';

	}

	/**
	 * Activation hook.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function activate() {

		$activate = filter_input( INPUT_POST, 'action', FILTER_UNSAFE_RAW );
		$checked  = filter_input( INPUT_POST, 'checked', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );

		if ( 'activate-selected' === $activate && count( $checked ) > 1 ) {
			return; // Do not redirect if bulk activating plugins.
		}

		set_transient( 'fuzion-admin-redirect', 5 * MINUTE_IN_SECONDS );

	}

}
