<?php
/**
 * The file that defines the AJAX plugin class
 *
 * All Ajax-related callbacks and functionality is registered here.
 *
 * @link https://getfuzion.io
 *
 * @package Fuzion
 * @subpackage Fuzion/App
 * @author Fuzion AI
 * @since 1.0.0
 */

namespace Fuzion\App;

use Fuzion\App\Models\Product;
use Fuzion\App\Models\SEO;
use Fuzion\App\Models\User;
use Exception;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The AJAX plugin class.
 *
 * @since 1.0.0
 */
class Ajax {

	/**
	 * API instance.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var Api\Auth $api;
	 */
	protected $api;

	/**
	 * Class constructor.
	 *
	 * @since 1.0.0
	 *
	 * @param Api\Auth $api  API instance.
	 */
	public function __construct( Api\Auth $api ) {

		if ( ! wp_doing_ajax() ) {
			return;
		}

		$this->api = $api;

		// Registration and API key actions.
		add_action( 'wp_ajax_ai_register', array( $this, 'register' ) );
		add_action( 'wp_ajax_ai_login', array( $this, 'login' ) );
		add_action( 'wp_ajax_ai_update_key', array( $this, 'update_key' ) );
		add_action( 'wp_ajax_ai_logout', array( $this, 'logout' ) );
		add_action( 'wp_ajax_ai_sync_stats', array( $this, 'sync_stats' ) );

		// Content AI actions.
		add_action( 'wp_ajax_ai_query', array( $this, 'ai_query' ) );
		// SEO actions.
		add_action( 'wp_ajax_seo_query', array( $this, 'seo_query' ) );
		// Image AI actions.
		add_action( 'wp_ajax_ai_image', array( $this, 'ai_image' ) );
		// Product Description actions.
		add_action( 'wp_ajax_ai_product_description', array( $this, 'product_description' ) );
		// Code AI actions.
		add_action( 'wp_ajax_ai_code', array( $this, 'ai_code' ) );

	}

	/**
	 * Check request permissions.
	 *
	 * @since 1.0.0
	 *
	 * @param string $type Capability to check.
	 *
	 * @return void
	 */
	private function check_permissions( string $type = 'manage_options' ) {

		check_ajax_referer( 'fuzion-nonce' );

		if ( ! current_user_can( $type ) ) {
			wp_send_json_error( esc_html__( 'Insufficient permissions', 'fuzion' ) );
		}

	}

	/**
	 * New user registration.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function register() {

		$this->check_permissions();

		$name    = filter_input( INPUT_POST, 'name', FILTER_UNSAFE_RAW );
		$email   = filter_input( INPUT_POST, 'email', FILTER_SANITIZE_EMAIL );
		$pass    = filter_input( INPUT_POST, 'password', FILTER_UNSAFE_RAW );
		$confirm = filter_input( INPUT_POST, 'password_confirmation', FILTER_UNSAFE_RAW );

		$args = array(
			'name'                  => sanitize_text_field( $name ),
			'email'                 => $email,
			'terms'                 => true,
			'password'              => htmlentities( $pass ),
			'password_confirmation' => htmlentities( $confirm ),
		);

		try {
			$response = $this->api->register( $args );
			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

	/**
	 * User login.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function login() {

		$this->check_permissions();

		$email = filter_input( INPUT_POST, 'email', FILTER_SANITIZE_EMAIL );
		$pass  = filter_input( INPUT_POST, 'password', FILTER_UNSAFE_RAW );

		$args = array(
			'email'    => $email,
			'password' => htmlentities( $pass ),
			'site'     => wp_parse_url( site_url(), PHP_URL_HOST ),
		);

		try {
			$response = $this->api->login( $args );
			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

	/**
	 * Update API key.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function update_key() {

		$this->check_permissions();

		$key = filter_input( INPUT_POST, 'key', FILTER_UNSAFE_RAW );
		$key = sanitize_text_field( $key );
		User::set_token( $key );

		try {
			$content_ai = new Api\Content();
			$response   = $content_ai->sync_stats();

			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

	/**
	 * Logout.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function logout() {

		$this->check_permissions();

		delete_option( 'fuzion-registered' );
		delete_option( 'fuzion-api-token' );
		delete_option( 'fuzion-credits' );

		wp_send_json_success();

	}

	/**
	 * Sync API data (stats and subscription).
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function sync_stats() {

		$this->check_permissions( 'edit_posts' );

		try {
			$content_ai = new Api\Content();
			$response   = $content_ai->sync_stats();

			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

	/**
	 * Handle API query to content AI.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function ai_query() {

		$this->check_permissions( 'edit_posts' );

		$prompt = filter_input( INPUT_POST, 'prompt', FILTER_UNSAFE_RAW );
		$type   = filter_input( INPUT_POST, 'type', FILTER_UNSAFE_RAW );

		// If prompt is empty - show an error.
		if ( empty( $prompt ) ) {
			wp_send_json_error( esc_html__( 'Please enter a prompt', 'fuzion' ) );
		}

		if ( 'heading' === $type ) { // Heading uses the same endpoint as title.
			$type = 'title';
		}

		// Optional fields.
		$credits     = filter_input( INPUT_POST, 'credits', FILTER_SANITIZE_NUMBER_INT );
		$temperature = filter_input( INPUT_POST, 'temperature', FILTER_SANITIZE_NUMBER_INT );
		$completions = filter_input( INPUT_POST, 'results', FILTER_SANITIZE_NUMBER_INT );

		$query = array(
			'completions' => $completions ?? 1,
			'max_tokens'  => $credits ?? 80,
			'prompt'      => htmlentities( $prompt ),
			'site'        => wp_parse_url( site_url(), PHP_URL_HOST ),
			'temperature' => $temperature ? $temperature / 100 : 0.6,
			'type'        => sanitize_text_field( $type ),
		);

		try {
			$content_ai = new Api\Content();
			$response   = $content_ai->query( $query );

			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

	/**
	 * Handle API query to SEO endpoint.
	 *
	 * @since 1.2.0
	 *
	 * @return void
	 */
	public function seo_query() {

		$this->check_permissions( 'edit_posts' );

		$keyword = filter_input( INPUT_POST, 'keyword', FILTER_UNSAFE_RAW );
		$keyword = sanitize_text_field( $keyword );

		$key  = Models\SEO::get_cache_key( $keyword );
		$data = get_transient( $key );

		if ( $data ) {
			wp_send_json_success( $data );
		}

		$query = array(
			'keywords' => $keyword,
			'site'     => wp_parse_url( site_url(), PHP_URL_HOST ),
		);

		try {
			$seo_ai   = new Api\SEO();
			$response = $seo_ai->query( $query );
			$response = SEO::parse_response( (array) $response, $keyword );

			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

	/**
	 * Handle API query to generate image.
	 *
	 * @since 1.3.0
	 *
	 * @return void
	 */
	public function ai_image() {

		$this->check_permissions( 'edit_posts' );

		$prompt  = filter_input( INPUT_POST, 'imagePrompt', FILTER_UNSAFE_RAW );
		$post_id = filter_input( INPUT_POST, 'postID', FILTER_SANITIZE_NUMBER_INT );

		$query = array(
			'prompt' => htmlentities( $prompt ),
			'site'   => wp_parse_url( site_url(), PHP_URL_HOST ),
		);

		try {
			$content_ai = new Api\Content();
			$response   = $content_ai->generate_image( $query );

			$attachment_id = Models\Image::save_image( $response->data, (int) $post_id, $query['prompt'] );
			$image_data    = wp_get_attachment_image_src( $attachment_id, 'large' );

			$response = array(
				'id'  => $attachment_id,
				'url' => $image_data[0],
			);

			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

	/**
	 * Handle API query to Product Description endpoint.
	 *
	 * @since 1.3.0
	 *
	 * @return void
	 */
	public function product_description() {

		$this->check_permissions( 'edit_posts' );

		$product_id = filter_input( INPUT_POST, 'product_id', FILTER_SANITIZE_NUMBER_INT );
		$product    = wc_get_product( $product_id );

		if ( ! $product ) {
			wp_send_json_error(
				array(
					'message' => __( 'Product not found', 'fuzion' ),
				),
				404
			);
		}

		$attributes = filter_input( INPUT_POST, 'attributes', FILTER_DEFAULT, FILTER_FORCE_ARRAY );
		$attributes = Product::get_product_attributes( $product, $attributes ?? array() );

		$tokens      = filter_input( INPUT_POST, 'credits', FILTER_SANITIZE_NUMBER_INT );
		$temperature = filter_input( INPUT_POST, 'temperature', FILTER_SANITIZE_NUMBER_INT );

		$query = array(
			'attributes'  => $attributes,
			'title'       => $product->get_title(),
			'site'        => wp_parse_url( site_url(), PHP_URL_HOST ),
			'max_tokens'  => $tokens ?? 80,
			'temperature' => $temperature ? $temperature / 100 : 0.6,
		);

		try {
			$product_description = new Api\WooCommerce\Product();
			$response            = $product_description->query( $query );

			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

	/**
	 * Handle API query to code AI endpoint.
	 *
	 * @since 1.4.0
	 *
	 * @return void
	 */
	public function ai_code() {

		$this->check_permissions( 'edit_posts' );

		$credits = filter_input( INPUT_POST, 'credits', FILTER_SANITIZE_NUMBER_INT );
		$prompt  = filter_input( INPUT_POST, 'prompt', FILTER_UNSAFE_RAW );
		$type    = filter_input( INPUT_POST, 'type', FILTER_UNSAFE_RAW );

		// If prompt is empty - show an error.
		if ( empty( $prompt ) ) {
			wp_send_json_error( esc_html__( 'Please enter a prompt inside the code block.', 'fuzion' ) );
		}

		$query = array(
			'max_tokens' => $credits ? (int) $credits : 300,
			'prompt'     => htmlentities( $prompt ),
			'site'       => wp_parse_url( site_url(), PHP_URL_HOST ),
		);

		if ( 'code_edit' === $type ) {
			$instructions = filter_input( INPUT_POST, 'instructions', FILTER_UNSAFE_RAW );

			if ( ! $instructions ) {
				wp_send_json_error( esc_html__( 'Prompt cannot be empty.', 'fuzion' ) );
			}

			/**
			 * Edits endpoint has different parameters:
			 * code - The input code to use as a starting point for the edit.
			 * prompt - The instruction that tells the model how to edit the prompt.
			 */
			$query['code']   = $query['prompt'];
			$query['prompt'] = htmlentities( $instructions );
		}

		try {
			$content_ai = new Api\Content();
			$response   = $content_ai->generate_code( $query, sanitize_text_field( $type ) );

			wp_send_json_success( $response );
		} catch ( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}

	}

}
