<?php
/**
 * Abstract API class
 *
 * This class defines all code necessary to communicate with the API.
 *
 * @link https://getfuzion.io
 *
 * @package Fuzion
 * @subpackage Fuzion/App/Api
 * @author Fuzion AI
 * @since 1.0.0
 */

namespace Fuzion\App\Api;

use Exception;
use Fuzion\App\Models\Credit;
use Fuzion\App\Models\User;
use stdClass;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Abstract Request class
 *
 * @since 1.0.0
 */
abstract class Request {

	/**
	 * API URL.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @var string
	 */
	protected $api_url = 'https://getfuzion.io/api/';

	/**
	 * Endpoint for API call.
	 *
	 * @since 1.0.0
	 * @access private
	 * @var string
	 */
	private $endpoint = '';

	/**
	 * Body for API call.
	 *
	 * @since 1.0.0
	 * @access private
	 * @var array
	 */
	private $request_body = array();

	/**
	 * Method used to do API call.
	 *
	 * @since 1.0.0
	 * @access private
	 * @var string
	 */
	private $method = 'POST';

	/**
	 * Request timeout.
	 *
	 * @since 1.3.0
	 * @access private
	 * @var int
	 */
	private $timeout = 10;

	/**
	 * Setter for $endpoint.
	 *
	 * @since 1.0.0
	 *
	 * @param string $endpoint  Endpoint.
	 *
	 * @return void
	 */
	protected function set_endpoint( string $endpoint ) {
		$this->endpoint = $endpoint;
	}

	/**
	 * Setter for $body.
	 *
	 * @since 1.0.0
	 *
	 * @param array $data  Array of data.
	 *
	 * @return void
	 */
	protected function set_request_body( array $data ) {
		$this->request_body = $data;
	}

	/**
	 * Setter for $method.
	 *
	 * @since 1.0.0
	 *
	 * @param string $method  Method.
	 *
	 * @return void
	 */
	protected function set_method( string $method ) {
		$this->method = $method;
	}

	/**
	 * Setter for $timeout.
	 *
	 * @since 1.3.0
	 *
	 * @param int $timeout  Timeout in seconds.
	 *
	 * @return void
	 */
	protected function set_timeout( int $timeout ) {
		$this->timeout = $timeout;
	}

	/**
	 * Get API URL.
	 *
	 * @since 1.1.0
	 *
	 * @return string
	 */
	private function get_url(): string {

		$url = $this->api_url;
		if ( defined( 'FUZION_API_URL' ) && FUZION_API_URL ) {
			$url = trailingslashit( FUZION_API_URL );
		}

		return $url . $this->endpoint;

	}

	/**
	 * Get arguments for request.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	private function get_args(): array {

		$header = array(
			'Accept'       => 'application/json',
			'Content-Type' => 'application/json',
		);

		$args = array(
			'method'  => $this->method,
			'timeout' => $this->timeout,
			'headers' => array_merge( $header, $this->add_auth_headers() ),
		);

		if ( ! empty( $this->request_body ) && 'POST' === $args['method'] ) {
			$args['body'] = $this->request_body;
		}

		return $args;

	}

	/**
	 * Do API request.
	 *
	 * @since 1.0.0
	 *
	 * @param array $params  Parameters to append to URL.
	 *
	 * @throws Exception  API exception.
	 *
	 * @return array
	 */
	protected function request( array $params = array() ): array {

		$url  = $this->get_url();
		$args = $this->get_args();

		if ( ! empty( $params ) ) {
			$url = add_query_arg( $params, $url );
		}

		if ( 'GET' === $args['method'] ) {
			$response = wp_remote_get( $url, $args );
		} elseif ( 'POST' === $args['method'] ) {
			$response = wp_remote_post( $url, $args );
		} else {
			throw new Exception( __( 'Unsupported API call method', 'fuzion' ) );
		}

		if ( is_wp_error( $response ) ) {
			throw new Exception( $response->get_error_message() );
		}

		return $response;

	}

	/**
	 * Add authorization headers.
	 *
	 * @since 1.2.0 Moved from Content class.
	 *
	 * @return array|string[]
	 */
	public function add_auth_headers(): array {

		$token = User::get_token();

		if ( empty( $token ) ) {
			return array();
		}

		return array(
			'Authorization' => 'Bearer ' . $token,
		);

	}

	/**
	 * Improve error messages.
	 *
	 * Filters the error messages from the API and makes them more user-friendly.
	 *
	 * @since 1.0.0
	 *
	 * @param string $message  Error message from API.
	 * @param string $context  Context.
	 *
	 * @return string
	 */
	protected function filter_error_response( string $message, string $context = '' ): string {

		// Invalid API key.
		if ( str_contains( $message, 'Unauthenticated' ) ) {
			$message = __( 'Expired or invalid API key. Please update your API key on the setting page.', 'fuzion' );
		}

		return $message;

	}

	/**
	 * Process API response.
	 *
	 * @param array $response  API response.
	 *
	 * @return stdClass
	 * @throws Exception  Response exception.
	 */
	protected function process_response( array $response ): stdClass {

		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );
		$body = json_decode( $body );

		if ( 200 === $code || 201 === $code ) {
			Credit::update_stats( $body );
			return $body;
		}

		if ( isset( $body->message ) ) {
			throw new Exception( $this->filter_error_response( $body->message ) );
		}

		throw new Exception( __( 'Error doing API call. Please try again.', 'fuzion' ) );

	}

}
