<?php
/**
 * Auth API class
 *
 * This class defines all code for registering, authenticating and managing API keys.
 *
 * @link https://getfuzion.io
 *
 * @package Fuzion
 * @subpackage Fuzion/App/Api
 * @author Fuzion AI
 * @since 1.0.0
 */

namespace Fuzion\App\Api;

use Fuzion\App\Models\User;
use Exception;
use stdClass;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Auth class
 *
 * @since 1.0.0
 */
class Auth extends Request {

	/**
	 * Add authorization headers.
	 *
	 * @since 1.2.0
	 *
	 * @return array|string[]
	 */
	public function add_auth_headers(): array {
		return array();
	}

	/**
	 * Register endpoint.
	 *
	 * @since 1.0.0
	 *
	 * @throws Exception  API exception.
	 *
	 * @param array $params  Parameters to append to URL.
	 *
	 * @return stdClass
	 */
	public function register( array $params ): stdClass {

		$this->set_method( 'POST' );
		$this->set_endpoint( 'register' );

		$response = $this->request( $params );

		// New user created.
		if ( 201 === wp_remote_retrieve_response_code( $response ) ) {
			// Make sure we mark that the registration was a success, user still needs to verify the email.
			User::set_registered();
			return new stdClass();
		}

		$body = wp_remote_retrieve_body( $response );
		return json_decode( $body );

	}

	/**
	 * Login endpoint.
	 *
	 * Authenticate and get API key. If email is not verified, user will not be able to authenticate.
	 *
	 * @since 1.0.0
	 *
	 * @throws Exception  API exception.
	 *
	 * @param array $params  Request parameters.
	 *
	 * @throws Exception API issues.
	 *
	 * @return stdClass
	 */
	public function login( array $params ): stdClass {

		$this->set_method( 'POST' );
		$this->set_endpoint( 'user/api-tokens' );

		$response = $this->request( $params );

		$code = wp_remote_retrieve_response_code( $response );

		$response = wp_remote_retrieve_body( $response );
		$response = json_decode( $response );

		// Token created and received.
		if ( 201 === $code && isset( $response->token ) ) {
			User::set_token( $response->token );
			return $response;
		}

		// Something is wrong.
		throw new Exception( $response->message );

	}

}
