<?php
/**
 * The file that defines the views for the admin side
 *
 * Everything admin view related - rendering.
 *
 * @link https://getfuzion.io
 *
 * @package Fuzion
 * @subpackage Fuzion/App/Admin
 * @author Fuzion AI
 * @since 1.0.0
 */

namespace Fuzion\App\Admin;

use Fuzion\App\Models\User;
use Fuzion\App\Traits\Utils;

/**
 * Abstract View class.
 *
 * @since 1.0.0
 */
abstract class View {

	use Utils;

	/**
	 * Render admin menu.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function render_page() {

		$view = $this->get_view();

		if ( empty( $view ) ) {
			if ( User::is_registered() && User::is_email_verified() && User::has_api_token() ) {
				$view = 'account';
			} elseif ( User::is_registered() && ! User::is_email_verified() ) {
				$view = 'confirm';
			} else {
				$view = 'login';
			}
		}

		$this->view( $view );

	}

	/**
	 * Render view.
	 *
	 * @since 1.0.0
	 *
	 * @param string $name  View ID. Corresponds to *.php file in views folder.
	 *
	 * @return void
	 */
	protected function view( string $name ) {

		$view = dirname( __DIR__ ) . "/views/$name.php";

		if ( ! file_exists( $view ) ) {
			return;
		}

		ob_start();
		include $view;
		echo ob_get_clean(); /* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped */

	}

	/**
	 * Get view page.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	protected function get_view(): string {

		$view = filter_input( INPUT_GET, 'view', FILTER_UNSAFE_RAW );

		return sanitize_text_field( $view );

	}

}
