<?php
/**
 * The file that defines the admin plugin class
 *
 * Everything admin related - registration, API key management.
 *
 * @link https://getfuzion.io
 *
 * @package Fuzion
 * @subpackage Fuzion/App/Admin
 * @author Fuzion AI
 * @since 1.0.0
 */

namespace Fuzion\App\Admin;

/**
 * Admin plugin class.
 *
 * @since 1.0.0
 */
class Admin extends View {

	/**
	 * Init hooks and actions.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function __construct() {

		if ( ! is_admin() ) {
			return;
		}

		add_action( 'admin_menu', array( $this, 'register_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_styles' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_meta_box_scripts' ) );

		add_action( 'admin_init', array( $this, 'maybe_redirect_to_plugin_page' ) );
		// Define the locale for this plugin for internationalization.
		add_action( 'admin_init', array( $this, 'load_plugin_textdomain' ) );
		// Add settings link to plugin page.
		add_filter( 'plugin_action_links_fuzion/fuzion.php', array( $this, 'settings_link' ) );

		add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );
		// Override user meta value of meta box order in product page.
		add_filter( 'get_user_option_meta-box-order_product', array( $this, 'sort_meta_box_order' ), 99 );

	}

	/**
	 * Register sub-menu under the WordPress "Settings" menu element.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function register_menu() {

		add_submenu_page(
			'options-general.php',
			__( 'Fuzion AI', 'fuzion' ),
			__( 'Fuzion AI', 'fuzion' ),
			'manage_options',
			$this->get_slug(),
			array( $this, 'render_page' )
		);

	}

	/**
	 * Load plugin styles.
	 *
	 * @since 1.0.0
	 *
	 * @param string $hook  The current admin page.
	 *
	 * @return void
	 */
	public function enqueue_styles( string $hook ) {

		// Run only on plugin pages.
		if ( 'settings_page_fuzion' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			$this->get_slug(),
			FUZION_DIR_URL . 'assets/css/ai-admin.min.css',
			array(),
			$this->get_version()
		);

	}


	/**
	 * Load plugin scripts.
	 *
	 * @since 1.0.0
	 *
	 * @param string $hook  The current admin page.
	 *
	 * @return void
	 */
	public function enqueue_scripts( string $hook ) {

		// Run only on plugin pages.
		if ( 'settings_page_fuzion' !== $hook ) {
			return;
		}

		wp_enqueue_script(
			$this->get_slug(),
			FUZION_DIR_URL . 'assets/js/ai-admin.min.js',
			array(),
			$this->get_version(),
			true
		);

		wp_localize_script(
			$this->get_slug(),
			'Fuzion',
			array(
				'nonce'  => wp_create_nonce( 'fuzion-nonce' ),
				'links'  => array(
					'pluginURL' => $this->get_url(),
				),
				'string' => array(
					'passwordsNotMatch' => esc_html__( 'Passwords do not match', 'fuzion' ),
					'statsSynced'       => esc_html__( 'Subscription data successfully synced', 'fuzion' ),
				),
			)
		);

	}

	/**
	 * Register localization for the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function load_plugin_textdomain() {

		load_plugin_textdomain(
			$this->get_slug(),
			false,
			$this->get_slug() . '/languages/'
		);

	}

	/**
	 * Add `Settings` link on the `Plugins` page.
	 *
	 * @since 1.0.0
	 *
	 * @param array $actions  Actions array.
	 *
	 * @return array
	 */
	public function settings_link( array $actions ): array {

		if ( ! current_user_can( 'manage_options' ) ) {
			return $actions;
		}

		$actions['fuzion-settings'] = '<a href="' . $this->get_url() . '" aria-label="' . esc_attr( __( 'Settings', 'fuzion' ) ) . '">' . esc_html__( 'Settings', 'fuzion' ) . '</a>';
		return $actions;

	}

	/**
	 * Maybe redirect to plugin page on activation.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function maybe_redirect_to_plugin_page() {

		if ( ! get_transient( 'fuzion-admin-redirect' ) ) {
			return;
		}

		delete_transient( 'fuzion-admin-redirect' );
		wp_safe_redirect( $this->get_url() );
		exit;

	}

	/**
	 * Add admin meta boxes.
	 *
	 * @since 1.3.0
	 *
	 * @param string $post_type Current post type.
	 */
	public function add_meta_boxes( string $post_type ) {

		add_meta_box(
			'fuzion_ai',
			__( 'Fuzion AI', 'fuzion' ),
			array( $this, 'render_product_meta_box' ),
			'product',
			'side',
			'core'
		);

	}

	/**
	 * Render product page fuzion ai options.
	 *
	 * @since 1.3.0
	 */
	public function render_product_meta_box() {
		$this->view( 'fuzion-ai' );
	}

	/**
	 * Sort product page metabox order.
	 *
	 * @since 1.3.0
	 *
	 * @param array $value User sorted meta box order.
	 *
	 * @return array $value Modified meta box order.
	 */
	public function sort_meta_box_order( $value ) {

		// WooCommerce makes their meta boxes high priority by saving product page meta box order to user meta.
		// We have overriden the user meta to bring our fuzion ai meta box to come after submit div.
		// We check if user has done any changes to meta box order before overriding the value.
		if ( is_array( $value ) && isset( $value['side'] ) && false === strpos( $value['side'], 'fuzion_ai' ) ) {
			$value['side'] = str_replace( 'submitdiv', 'submitdiv,fuzion_ai', $value['side'] );
		}

		return $value;

	}

	/**
	 * Enqueue ai meta box style and scripts.
	 *
	 * @since 1.3.0
	 *
	 * @param string $hook  The current admin page.
	 *
	 * @return void
	 */
	public function enqueue_meta_box_scripts( string $hook ) {

		if ( 'post.php' !== $hook ) {
			return;
		}

		$screen = get_current_screen();
		if ( 'product' !== $screen->id ) {
			return;
		}

		wp_enqueue_style(
			'fuzion-ai-metabox-style',
			FUZION_DIR_URL . 'assets/css/ai-metabox.min.css',
			array(),
			$this->get_version()
		);

		wp_enqueue_script(
			'fuzion-ai-metabox-script',
			FUZION_DIR_URL . 'assets/js/ai-metabox.min.js',
			array( 'jquery' ),
			$this->get_version(),
			true
		);

		wp_localize_script(
			'fuzion-ai-metabox-script',
			'Fuzion',
			array(
				'nonce'  => wp_create_nonce( 'fuzion-nonce' ),
				'string' => array(
					'titleIsRequired' => esc_html__( 'Fuzion AI Error: To generate the product description a product title is required.', 'fuzion' ),
				),
			)
		);

	}
}
