<?php
/**
 * Plugin Name: FukuroChat Connector
 * Plugin URI: https://www.fukurochat.com
 * Description: Connect your WooCommerce store to FukuroChat and automatically sync your products with your AI assistant. Includes conversion tracking and analytics.
 * Version: 1.0.10
 * Author: FukuroChat
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Fukurochat_Connector {

    public function __construct() {
        // Main hooks

        // Hook to add settings page
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));

        // Hook to inject FukuroChat widget
        add_action('wp_footer', array($this, 'inject_fukurochat_widget'));

        // Hook to add Customizer settings
        add_action('customize_register', array($this, 'customize_register'));

        // Hooks for automatic product sync
        add_action('woocommerce_new_product', array($this, 'sync_product_create'), 10, 1);
        add_action('woocommerce_update_product', array($this, 'sync_product_update'), 10, 1);
        add_action('wp_trash_post', array($this, 'sync_product_trash'), 10, 1);
        add_action('before_delete_post', array($this, 'sync_product_delete'), 10, 1);
        
        // Hook for draft/publish status changes
        add_action('transition_post_status', array($this, 'sync_product_status_change'), 10, 3);

        // Hook to show admin notices
        add_action('admin_notices', array($this, 'show_sync_notices'));

        // Hooks for conversion tracking
        add_action('woocommerce_new_order', array($this, 'track_order_conversion'), 10, 1);
        add_action('woocommerce_thankyou', array($this, 'track_order_conversion'), 10, 1);
        add_action('wp_footer', array($this, 'inject_conversion_tracking_script'));
        add_action('woocommerce_checkout_create_order', array($this, 'save_chatbot_meta_to_order'), 10, 2);
        add_action('woocommerce_store_api_checkout_order_processed', array($this, 'save_chatbot_meta_to_order_api'), 10, 1);
        
        // AJAX endpoint to track add-to-cart events
        add_action('wp_ajax_fukurochat_track_add_to_cart', array($this, 'ajax_track_add_to_cart'));
        add_action('wp_ajax_nopriv_fukurochat_track_add_to_cart', array($this, 'ajax_track_add_to_cart'));

        // AJAX endpoint for OAuth-style connection
        add_action('wp_ajax_fukurochat_exchange_token', array($this, 'ajax_exchange_token'));
        
        // AJAX endpoint for toggling sync
        add_action('wp_ajax_fukurochat_toggle_sync', array($this, 'ajax_toggle_sync'));
        
        // AJAX endpoint for initial sync
        add_action('wp_ajax_fukurochat_initial_sync', array($this, 'ajax_initial_sync'));

        // Add custom metabox for alternative products
        add_action('add_meta_boxes', array($this, 'add_alternative_products_metabox'));
        add_action('save_post', array($this, 'save_alternative_products_field'));

        // Flush rewrite rules on activation
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    public function add_admin_menu() {
        add_menu_page(
            'FukuroChat Configuration',    // Page title
            'FukuroChat',                  // Menu text
            'manage_options',              // Required capability
            'fukurochat-connector',        // Menu slug
            array($this, 'render_admin_page'), // Render function
            plugin_dir_url(__FILE__) . 'icon-256x256.png', // Menu icon
            58                             // Menu position
        );

        // Rename first submenu to "Configuration"
        add_submenu_page(
            'fukurochat-connector',        // Parent slug
            'Configuration',               // Page title
            'Configuration',               // Menu text
            'manage_options',              // Required capability
            'fukurochat-connector',        // Menu slug (same as parent to override)
            array($this, 'render_admin_page') // Render function
        );

        // Add Analytics submenu
        add_submenu_page(
            'fukurochat-connector',        // Parent slug
            'Analytics',                   // Page title
            'Analytics',                   // Menu text
            'manage_options',              // Required capability
            'fukurochat-analytics',        // Menu slug
            array($this, 'render_analytics_page') // Render function
        );
        
        // Hook to enqueue admin styles and scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }

    public function register_settings() {
        register_setting(
            'fukurochat_settings',          // Options group
            'fukurochat_assistant_uuid',    // Option name
            array($this, 'validate_assistant_uuid') // Validation function
        );

        register_setting(
            'fukurochat_settings',          // Options group
            'fukurochat_bubble_text',       // Option name
            'sanitize_text_field'           // Validation function
        );

        register_setting(
            'fukurochat_settings',          // Options group
            'fukurochat_target_id',         // Option name
            'sanitize_text_field'           // Validation function
        );

        register_setting(
            'fukurochat_settings',          // Options group
            'fukurochat_position',          // Option name
            'sanitize_text_field'           // Validation function
        );

        register_setting(
            'fukurochat_settings',          // Options group
            'fukurochat_offset_horizontal', // Option name
            'absint'                        // Validation function
        );

        register_setting(
            'fukurochat_settings',          // Options group
            'fukurochat_offset_vertical',   // Option name
            'absint'                        // Validation function
        );

        register_setting(
            'fukurochat_settings',          // Options group
            'fukurochat_sync_enabled',      // Option name
            array($this, 'validate_sync_enabled') // Validation function
        );

        register_setting(
            'fukurochat_settings',          // Options group
            'fukurochat_api_secret',        // Option name
            array($this, 'validate_api_secret') // Validation function
        );

        // No need for settings sections and fields since we handle the form directly
    }

    public function enqueue_admin_assets($hook) {
        // Register and enqueue a custom style handle for our inline styles
        wp_register_style('fukurochat-admin', false, array(), '1.0.7');
        wp_enqueue_style('fukurochat-admin');
        
        // Icon styles for all admin pages
        wp_add_inline_style('fukurochat-admin', '
            #toplevel_page_fukurochat-connector .wp-menu-image img {
                width: 20px;
                height: 20px;
                padding: 6px 0;
            }
        ');

        // Only load page-specific styles on our plugin page
        if ($hook !== 'toplevel_page_fukurochat-connector') {
            return;
        }

        // Enqueue page styles
        wp_add_inline_style('fukurochat-admin', '
            .fukuro-step {
                background: #f9f9f9;
                border: 1px solid #ddd;
                border-radius: 8px;
                padding: 20px;
                margin-bottom: 20px;
            }
            .fukuro-step h2 {
                margin-top: 0;
                color: #333;
            }
            .fukuro-button {
                background: #333;
                color: white !important;
                padding: 8px 16px;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                text-decoration: none !important;
                display: inline-block;
                margin: 10px 0;
            }
            .fukuro-button:hover {
                background: #555;
                color: white !important;
            }
            .fukuro-info {
                background: #e7f3ff;
                border-left: 4px solid #0073aa;
                padding: 12px;
                margin: 10px 0;
            }
            .fukuro-uuid-field {
                width: 100%;
                max-width: 400px;
                padding: 8px;
                font-family: monospace;
            }
            /* Toggle Switch Styles */
            .fukuro-toggle-container {
                display: flex;
                align-items: center;
                gap: 10px;
                margin: 15px 0;
            }
            .fukuro-toggle {
                position: relative;
                display: inline-block;
                width: 50px;
                height: 24px;
            }
            .fukuro-toggle input {
                opacity: 0;
                width: 0;
                height: 0;
            }
            .fukuro-toggle-slider {
                position: absolute;
                cursor: pointer;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: #ccc;
                transition: .4s;
                border-radius: 24px;
            }
            .fukuro-toggle-slider:before {
                position: absolute;
                content: "";
                height: 18px;
                width: 18px;
                left: 3px;
                bottom: 3px;
                background-color: white;
                transition: .4s;
                border-radius: 50%;
            }
            .fukuro-toggle input:checked + .fukuro-toggle-slider {
                background-color: #2196F3;
            }
            .fukuro-toggle input:disabled + .fukuro-toggle-slider {
                background-color: #e0e0e0;
                cursor: not-allowed;
            }
            .fukuro-toggle input:checked + .fukuro-toggle-slider:before {
                transform: translateX(26px);
            }
            .fukuro-toggle-label {
                font-size: 14px;
                color: #666;
            }
            .fukuro-toggle-label.disabled {
                color: #999;
            }
            .fukuro-connect-button {
                background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
                color: white !important;
                padding: 20px 40px;
                border: none;
                border-radius: 8px;
                cursor: pointer;
                font-size: 18px;
                font-weight: 600;
                text-decoration: none !important;
                display: inline-flex;
                align-items: center;
                gap: 12px;
                box-shadow: 0 4px 15px rgba(0, 123, 255, 0.4);
                transition: all 0.3s ease;
            }
            .fukuro-connect-button:hover {
                transform: translateY(-2px);
                box-shadow: 0 6px 20px rgba(0, 123, 255, 0.6);
                color: white !important;
            }
            .fukuro-connected-info {
                background: #e7f9f0;
                border-left: 4px solid #10b981;
                padding: 20px;
                margin: 20px 0;
                border-radius: 4px;
            }
            .fukuro-disconnect-button {
                background: #ef4444;
                color: white !important;
                padding: 8px 16px;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                font-size: 14px;
            }
            .fukuro-disconnect-button:hover {
                background: #dc2626;
            }
        ');

        // Enqueue page scripts - jQuery is already loaded in admin
        wp_add_inline_script('jquery', '
            jQuery(document).ready(function($) {
                // No longer needed since we removed the conditional field display
            });
        ');

        // Enqueue customizer scripts
        if ($hook === 'customize.php' || is_customize_preview()) {
            wp_add_inline_script('customize-controls', '
                (function($) {
                    wp.customize.bind("ready", function() {
                        var positionControl = wp.customize.control("fukurochat_position");
                        var horizontalControl = wp.customize.control("fukurochat_offset_horizontal");
                        var verticalControl = wp.customize.control("fukurochat_offset_vertical");

                        function updateOffsetLabels() {
                            var position = wp.customize("fukurochat_position").get();
                            var hLabel = "Horizontal Offset (px)";
                            var vLabel = "Vertical Offset (px)";
                            var hDesc = "Distance from the side edge";
                            var vDesc = "Distance from the top/bottom edge";
                            var showH = true;
                            var showV = true;

                            switch(position) {
                                case "bottom-right":
                                    hLabel = "Distance from Right (px)";
                                    vLabel = "Distance from Bottom (px)";
                                    hDesc = "Horizontal spacing from the right edge";
                                    vDesc = "Vertical spacing from the bottom edge";
                                    break;
                                case "bottom-left":
                                    hLabel = "Distance from Left (px)";
                                    vLabel = "Distance from Bottom (px)";
                                    hDesc = "Horizontal spacing from the left edge";
                                    vDesc = "Vertical spacing from the bottom edge";
                                    break;
                                case "center-right":
                                    hLabel = "Distance from Right (px)";
                                    hDesc = "Horizontal spacing from the right edge";
                                    showV = false;
                                    break;
                                case "center-left":
                                    hLabel = "Distance from Left (px)";
                                    hDesc = "Horizontal spacing from the left edge";
                                    showV = false;
                                    break;
                                case "top-right":
                                    hLabel = "Distance from Right (px)";
                                    vLabel = "Distance from Top (px)";
                                    hDesc = "Horizontal spacing from the right edge";
                                    vDesc = "Vertical spacing from the top edge";
                                    break;
                                case "top-left":
                                    hLabel = "Distance from Left (px)";
                                    vLabel = "Distance from Top (px)";
                                    hDesc = "Horizontal spacing from the left edge";
                                    vDesc = "Vertical spacing from the top edge";
                                    break;
                            }

                            // Update labels and descriptions
                            if (horizontalControl) {
                                horizontalControl.container.find(".customize-control-title").text(hLabel);
                                horizontalControl.container.find(".description").text(hDesc);
                                horizontalControl.container.toggle(showH);
                            }

                            if (verticalControl) {
                                verticalControl.container.find(".customize-control-title").text(vLabel);
                                verticalControl.container.find(".description").text(vDesc);
                                verticalControl.container.toggle(showV);
                            }
                        }

                        // Update on position change
                        wp.customize("fukurochat_position", function(setting) {
                            setting.bind(updateOffsetLabels);
                        });

                        // Initial update
                        updateOffsetLabels();
                    });
                })(jQuery);
            ');
        }
    }

    public function validate_assistant_uuid($input) {
        $input = sanitize_text_field($input);

        // Allow empty value to disable the widget
        if (empty($input)) {
            add_settings_error(
                'fukurochat_assistant_uuid',
                'settings_updated',
                'Widget disabled. The Assistant UUID has been cleared.',
                'success'
            );
            return '';
        }

        // Validate it has exactly 36 characters
        if (strlen($input) !== 36) {
            add_settings_error(
                'fukurochat_assistant_uuid',
                'invalid_length',
                'The Assistant UUID must be exactly 36 characters long or empty to disable.',
                'error'
            );
            return get_option('fukurochat_assistant_uuid');
        }

        // Validate UUID format (8-4-4-4-12)
        if (!preg_match('/^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/i', $input)) {
            add_settings_error(
                'fukurochat_assistant_uuid',
                'invalid_format',
                'The Assistant UUID format is invalid. It must follow the format: 8-4-4-4-12 hexadecimal characters.',
                'error'
            );
            return get_option('fukurochat_assistant_uuid');
        }

        // If everything is valid, return the sanitized input
        // WordPress will show its default success message
        return $input;
    }

    public function validate_sync_enabled($input) {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is verified by Settings API
        $api_secret = isset($_POST['fukurochat_api_secret']) ? sanitize_text_field(wp_unslash($_POST['fukurochat_api_secret'])) : '';
        
        // If trying to enable sync, validate credentials
        if ($input) {
            if (empty($api_secret)) {
                add_settings_error(
                    'fukurochat_sync_enabled',
                    'missing_api_secret',
                    'API Secret is required to enable automatic sync.',
                    'error'
                );
                return false; // Don't enable sync
            }
            
            // Check if credentials have been validated
            $credentials_status = get_option('fukurochat_credentials_valid', null);
            if ($credentials_status === null || !is_array($credentials_status) || !$credentials_status['valid']) {
                add_settings_error(
                    'fukurochat_sync_enabled',
                    'credentials_not_validated',
                    'Your API Secret is not valid. Sync cannot be enabled.',
                    'error'
                );
                return false; // Don't enable sync
            }
            
            // Credentials are valid, allow enabling sync
            add_settings_error(
                'fukurochat_sync_enabled',
                'sync_enabled',
                'Product sync has been enabled successfully!',
                'success'
            );
        }
        
        return (bool) $input;
    }

    public function validate_api_secret($input) {
        $input = sanitize_text_field($input);
        
        // Only validate length if API Secret is provided
        if (!empty($input) && strlen($input) !== 64) {
            add_settings_error(
                'fukurochat_api_secret',
                'invalid_length',
                'API Secret must be exactly 64 characters long.',
                'error'
            );
            // Clear credentials validation status
            delete_option('fukurochat_credentials_valid');
            return get_option('fukurochat_api_secret', '');
        }
        
        // If API Secret is provided and valid length, validate with API
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce is verified by Settings API
        $uuid = isset($_POST['fukurochat_assistant_uuid']) ? sanitize_text_field(wp_unslash($_POST['fukurochat_assistant_uuid'])) : '';
        
        if (!empty($input) && !empty($uuid)) {
            $validation_result = $this->validate_credentials_with_api($uuid, $input);
            
            if ($validation_result['valid']) {
                // Save validation status
                update_option('fukurochat_credentials_valid', array(
                    'valid' => true,
                    'checked_at' => current_time('mysql'),
                    'quota_data' => $validation_result['data']
                ));
                
                add_settings_error(
                    'fukurochat_api_secret',
                    'credentials_valid',
                    'API Secret validated successfully! You can now enable product sync.',
                    'success'
                );
            } else {
                // Save error status
                update_option('fukurochat_credentials_valid', array(
                    'valid' => false,
                    'checked_at' => current_time('mysql'),
                    'error' => $validation_result['error']
                ));
                
                add_settings_error(
                    'fukurochat_api_secret',
                    'credentials_invalid',
                    'Invalid API Secret: ' . $validation_result['error'],
                    'error'
                );
            }
        } elseif (empty($input)) {
            // Clear validation status if API Secret is removed
            delete_option('fukurochat_credentials_valid');
        }
        
        return $input;
    }

    // Validate credentials with FukuroChat API
    private function validate_credentials_with_api($uuid, $secret) {
        $api_url = 'https://app.fukurochat.com/api/index.php';
        
        $response = wp_remote_post($api_url, array(
            'timeout' => 15,
            'body' => array(
                'assistantUUID' => $uuid,
                'secret' => $secret,
                'action' => 'get_quota',
                'source' => 'woocommerce'
            )
        ));

        if (is_wp_error($response)) {
            return array(
                'valid' => false,
                'error' => 'Connection error: ' . $response->get_error_message()
            );
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        // Check API response format
        if (!isset($data['status'])) {
            return array(
                'valid' => false,
                'error' => 'Invalid API response format'
            );
        }

        if ($data['status'] === false) {
            return array(
                'valid' => false,
                'error' => !empty($data['error']) ? $data['error'] : 'Unknown error from API'
            );
        }

        // Credentials are valid
        return array(
            'valid' => true,
            'data' => $data['data']
        );
    }

    public function customize_register($wp_customize) {
        // Add FukuroChat Section
        $wp_customize->add_section('fukurochat_widget', array(
            'title'       => __('FukuroChat Widget', 'fukurochat-connector'),
            'description' => __('Configure your Assistant UUID in the FukuroChat Connector app settings. Then customize the optional settings below.', 'fukurochat-connector'),
            'priority'    => 160,
        ));

        // Setting: Enable/Disable Widget
        $wp_customize->add_setting('fukurochat_widget_enabled', array(
            'default'           => true,
            'transport'         => 'refresh',
            'sanitize_callback' => 'wp_validate_boolean',
        ));

        $wp_customize->add_control('fukurochat_widget_enabled', array(
            'label'       => __('Enable FukuroChat Widget', 'fukurochat-connector'),
            'description' => __('Toggle to enable or disable the FukuroChat widget on your site', 'fukurochat-connector'),
            'section'     => 'fukurochat_widget',
            'type'        => 'checkbox',
            'priority'    => 10,
        ));

        // Setting: Bubble Text
        $wp_customize->add_setting('fukurochat_bubble_text', array(
            'default'           => '',
            'transport'         => 'refresh',
            'sanitize_callback' => 'sanitize_text_field',
        ));

        $wp_customize->add_control('fukurochat_bubble_text', array(
            'label'       => __('Bubble Text', 'fukurochat-connector'),
            'description' => __('e.g., Chat with us', 'fukurochat-connector'),
            'section'     => 'fukurochat_widget',
            'type'        => 'text',
            'priority'    => 20,
        ));

        // Setting: Target Element ID
        $wp_customize->add_setting('fukurochat_target_id', array(
            'default'           => '',
            'transport'         => 'refresh',
            'sanitize_callback' => 'sanitize_text_field',
        ));

        $wp_customize->add_control('fukurochat_target_id', array(
            'label'       => __('Target Element ID', 'fukurochat-connector'),
            'description' => __('e.g., chat-widget-container', 'fukurochat-connector'),
            'section'     => 'fukurochat_widget',
            'type'        => 'text',
            'priority'    => 30,
        ));

        // Setting: Widget Position
        $wp_customize->add_setting('fukurochat_position', array(
            'default'           => 'bottom-right',
            'transport'         => 'refresh',
            'sanitize_callback' => 'sanitize_text_field',
        ));

        $wp_customize->add_control('fukurochat_position', array(
            'label'       => __('Widget Position', 'fukurochat-connector'),
            'description' => __('Choose where the chat widget appears on your site', 'fukurochat-connector'),
            'section'     => 'fukurochat_widget',
            'type'        => 'select',
            'choices'     => array(
                'bottom-right' => __('Bottom Right', 'fukurochat-connector'),
                'bottom-left'  => __('Bottom Left', 'fukurochat-connector'),
                'center-right' => __('Center Right', 'fukurochat-connector'),
                'center-left'  => __('Center Left', 'fukurochat-connector'),
                'top-right'    => __('Top Right', 'fukurochat-connector'),
                'top-left'     => __('Top Left', 'fukurochat-connector'),
            ),
            'priority'    => 40,
        ));

        // Setting: Horizontal Offset
        $wp_customize->add_setting('fukurochat_offset_horizontal', array(
            'default'           => 24,
            'transport'         => 'refresh',
            'sanitize_callback' => 'absint',
        ));

        $wp_customize->add_control('fukurochat_offset_horizontal', array(
            'label'       => __('Horizontal Offset (px)', 'fukurochat-connector'),
            'description' => __('Distance from the side edge', 'fukurochat-connector'),
            'section'     => 'fukurochat_widget',
            'type'        => 'number',
            'input_attrs' => array(
                'min'  => 0,
                'max'  => 500,
                'step' => 1,
            ),
            'priority'    => 50,
        ));

        // Setting: Vertical Offset
        $wp_customize->add_setting('fukurochat_offset_vertical', array(
            'default'           => 24,
            'transport'         => 'refresh',
            'sanitize_callback' => 'absint',
        ));

        $wp_customize->add_control('fukurochat_offset_vertical', array(
            'label'       => __('Vertical Offset (px)', 'fukurochat-connector'),
            'description' => __('Distance from the top/bottom edge', 'fukurochat-connector'),
            'section'     => 'fukurochat_widget',
            'type'        => 'number',
            'input_attrs' => array(
                'min'  => 0,
                'max'  => 500,
                'step' => 1,
            ),
            'priority'    => 60,
        ));

        // Add JavaScript to control visibility of offset fields
        $wp_customize->add_control(
            new WP_Customize_Control(
                $wp_customize,
                'fukurochat_offset_script',
                array(
                    'section'  => 'fukurochat_widget',
                    'type'     => 'hidden',
                    'priority' => 70,
                )
            )
        );
    }

    public function show_sync_notices() {
        // Show on product edit pages and product list pages
        $screen = get_current_screen();
        if (!$screen || ($screen->id !== 'product' && $screen->id !== 'edit-product')) {
            return;
        }

        $last_sync = get_transient('fukurochat_last_sync_result');
        if ($last_sync) {
            if ($last_sync['success']) {
                echo '<div class="notice notice-success is-dismissible"><p><strong>FukuroChat:</strong> ' . esc_html($last_sync['message']) . '</p></div>';
            } else {
                echo '<div class="notice notice-error is-dismissible"><p><strong>FukuroChat:</strong> Sync failed. ' . esc_html($last_sync['message']) . '</p></div>';
            }
            delete_transient('fukurochat_last_sync_result');
        }
    }

    public function render_admin_page() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $uuid = get_option('fukurochat_assistant_uuid', '');
        $assistant_name = get_option('fukurochat_assistant_name', '');
        $sync_enabled = get_option('fukurochat_sync_enabled', '0');
        $is_connected = !empty($uuid);
        ?>
        <div class="wrap">
            <h1>
                <img src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'icon-256x256.png'); ?>" alt="FukuroChat" style="width: 32px; height: 32px; vertical-align: middle; margin-right: 10px;">
                <?php echo esc_html(get_admin_page_title()); ?>
            </h1>

            <?php settings_errors(); ?>

            <p style="margin-bottom: 30px;">
                This plugin connects your WooCommerce store with <a href="https://www.fukurochat.com" target="_blank">FukuroChat</a>, 
                an AI-powered chatbot assistant that helps your customers find products, answer questions, and improve conversions. 
                Follow the 3 steps below to configure your assistant and start providing intelligent customer support.
            </p>

            <!-- Two column layout -->
            <div style="display: flex; gap: 20px; align-items: flex-start;">
                <!-- Left column: Steps -->
                <div style="flex: 1;">
                    <!-- Step 1: Configure Assistant -->
                    <div class="fukuro-step">
                        <h2>Step 1: Connect your Assistant</h2>
                        
                        <?php if (!$is_connected): ?>
                            <p>Click the button below to connect your FukuroChat assistant. You'll be able to select from your existing assistants.</p>
                            <button type="button" class="fukuro-connect-button" id="fukuro-connect-btn">
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"></path>
                                    <path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"></path>
                                </svg>
                                Connect with FukuroChat
                            </button>
                        <?php else: ?>
                            <div class="fukuro-connected-info">
                                <p style="margin: 0 0 10px 0;">
                                    <strong>✓ Connected to:</strong> <?php echo esc_html($assistant_name ? $assistant_name : 'FukuroChat Assistant'); ?>
                                </p>
                                <p style="margin: 0; font-family: monospace; font-size: 12px; color: #666;">
                                    UUID: <?php echo esc_html($uuid); ?>
                                </p>
                            </div>
                            
                            <button type="button" class="fukuro-disconnect-button" id="fukuro-disconnect-btn" style="margin-top: 15px;">
                                Disconnect Assistant
                            </button>
                        <?php endif; ?>
                    </div>

            <?php if ($is_connected): ?>
            <!-- Step 2: Enable Automatic Sync -->
            <div class="fukuro-step">
                <h2>Step 2: Enable Automatic Product Sync</h2>
                <p>Turn on automatic synchronization to keep your products up to date with your FukuroChat assistant.</p>
                
                <div class="fukuro-toggle-container" style="margin-top: 15px;">
                    <label class="fukuro-toggle">
                        <input 
                            type="checkbox" 
                            id="fukurochat_sync_enabled"
                            <?php checked($sync_enabled, '1'); ?>
                        />
                        <span class="fukuro-toggle-slider"></span>
                    </label>
                    <span class="fukuro-toggle-label">
                        <strong>Enable Automatic Product Sync</strong><br>
                        <small style="color: #666;">
                            When enabled, products will be automatically synchronized with your FukuroChat assistant
                        </small>
                    </span>
                </div>
            </div>

            <!-- Step 3: Enable Widget -->
            <div class="fukuro-step">
                <h2>Step 3: Enable the widget in your theme</h2>
                <ol>
                    <li>Click the button below to open the Theme Customizer</li>
                    <li>Look for the "FukuroChat Widget" section</li>
                    <li>Enable the widget and customize optional settings</li>
                    <li>Click "Publish" to save your changes - the widget will appear automatically</li>
                </ol>
                
                <a href="<?php echo esc_url(admin_url('customize.php?autofocus[section]=fukurochat_widget&return=' . urlencode(admin_url('admin.php?page=fukurochat-connector')))); ?>" class="fukuro-button" target="_blank">
                    Open Theme Customizer
                </a>
                
                <div class="fukuro-info">
                    <strong>ℹ️ Additional widget options in the theme customizer:</strong><br>
                    <strong>Bubble Text</strong> - Text to display next to the chat icon (optional)<br>
                    <strong>Widget Position</strong> - Choose where the widget appears: bottom-right, bottom-left, top-right, top-left, center-right, or center-left<br>
                    <strong>Horizontal & Vertical Offset</strong> - Adjust distance from screen edges in pixels<br>
                    <strong>Target Element ID</strong> - Custom placement for embedding the widget in a specific element (optional)
                </div>
            </div>

            <?php endif; ?>
                </div><!-- End left column -->

                <!-- Right column: Need Help? -->
                <div style="flex: 0 0 300px;">
                    <div style="background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px; padding: 15px;">
                        <h3 style="margin-top: 0; font-size: 14px;">Need Help?</h3>
                        <p style="margin: 10px 0; font-size: 13px;">
                            Visit <a href="https://www.fukurochat.com" target="_blank">FukuroChat Documentation</a> for more information about setting up your assistant.
                        </p>
                        <p style="margin: 10px 0; font-size: 13px;">
                            <strong>Requirements:</strong><br>
                            You must have an active account at <a href="https://www.fukurochat.com" target="_blank">fukurochat.com</a> to use this plugin.
                        </p>
                    </div>
                </div><!-- End right column -->
            </div><!-- End two column layout -->
            
            <script>
            var ajaxurl = '<?php echo esc_js(admin_url('admin-ajax.php')); ?>';
            
            jQuery(document).ready(function($) {
                var connectPopup = null;
                
                // Connect button handler
                $('#fukuro-connect-btn').on('click', function() {
                    var width = 600;
                    var height = 700;
                    var left = (screen.width - width) / 2;
                    var top = (screen.height - height) / 2;
                    var currentUrl = '<?php echo esc_js(admin_url('admin.php?page=fukurochat-connector')); ?>';
                    
                    connectPopup = window.open(
                        'https://app.fukurochat.com/webservices/connector/?origin=' + encodeURIComponent(window.location.origin),
                        'fukurochat_connect',
                        'width=' + width + ',height=' + height + ',left=' + left + ',top=' + top + ',resizable=yes,scrollbars=yes,location=no,toolbar=no'
                    );
                    
                    if (!connectPopup) {
                        alert('Please allow popups for this site to connect with FukuroChat');
                        return;
                    }
                });
                
                // Listen for postMessage from popup
                window.addEventListener('message', function(event) {
                    // Validate origin
                    if (event.origin !== 'https://app.fukurochat.com') {
                        return;
                    }
                    
                    // Check message type
                    if (event.data && event.data.type === 'fukurochat_connection') {
                        var token = event.data.token;
                        
                        if (!token) {
                            alert('Invalid connection token received');
                            return;
                        }
                        
                        // Close popup
                        if (connectPopup) {
                            connectPopup.close();
                        }
                        
                        // Show loading state
                        $('#fukuro-connect-btn').prop('disabled', true).html('Connecting...');
                        
                        // Exchange token for credentials
                        $.ajax({
                            url: ajaxurl,
                            method: 'POST',
                            data: {
                                action: 'fukurochat_exchange_token',
                                token: token,
                                nonce: '<?php echo esc_js(wp_create_nonce('fukurochat_connect')); ?>'
                            },
                            success: function(response) {
                                if (response.success) {
                                    location.reload();
                                } else {
                                    alert('Connection failed: ' + response.data);
                                    $('#fukuro-connect-btn').prop('disabled', false).html('<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"></path><path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"></path></svg> Connect with FukuroChat');
                                }
                            },
                            error: function() {
                                alert('Connection failed. Please try again.');
                                $('#fukuro-connect-btn').prop('disabled', false).html('<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"></path><path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"></path></svg> Connect with FukuroChat');
                            }
                        });
                    }
                });
                
                // Disconnect button handler
                $('#fukuro-disconnect-btn').on('click', function() {
                    if (confirm('Are you sure you want to disconnect this assistant? The widget will stop working until you connect a new assistant.')) {
                        window.location.href = '<?php echo esc_js(admin_url('admin.php?page=fukurochat-connector')); ?>';
                        // Clear the UUID by submitting empty form
                        var nonceField = '<input type="hidden" name="_wpnonce" value="<?php echo esc_js(wp_create_nonce('fukurochat_settings-options')); ?>">';
                        var refererField = '<input type="hidden" name="_wp_http_referer" value="<?php echo esc_js(admin_url('admin.php?page=fukurochat-connector')); ?>">';
                        $('<form method="post" action="options.php">' +
                            nonceField +
                            refererField +
                            '<input type="hidden" name="option_page" value="fukurochat_settings">' +
                            '<input type="hidden" name="action" value="update">' +
                            '<input type="hidden" name="fukurochat_assistant_uuid" value="">' +
                            '<input type="hidden" name="fukurochat_api_secret" value="">' +
                            '<input type="hidden" name="fukurochat_sync_enabled" value="0">' +
                            '</form>').appendTo('body').submit();
                    }
                });
                
                // Sync toggle handler - save on change
                $('#fukurochat_sync_enabled').on('change', function() {
                    var isEnabled = $(this).is(':checked');
                    var $toggle = $(this);
                    var $container = $toggle.closest('.fukuro-toggle-container');
                    
                    // If enabling sync, show confirmation
                    if (isEnabled) {
                        if (!confirm('This will start the initial synchronization of all published products with your FukuroChat assistant. Do you want to continue?')) {
                            $toggle.prop('checked', false);
                            return;
                        }
                    }
                    
                    // Remove any existing messages
                    $container.find('.fukuro-sync-message').remove();
                    
                    // Show saving message
                    var $saving = $('<div class="fukuro-sync-message" style="margin-top: 10px; padding: 8px 12px; background: #f0f6fc; border: 1px solid #0969da; border-radius: 4px; color: #0969da; font-size: 12px;">💾 Saving...</div>');
                    $container.append($saving);
                    
                    $.ajax({
                        url: ajaxurl,
                        method: 'POST',
                        data: {
                            action: 'fukurochat_toggle_sync',
                            enabled: isEnabled ? '1' : '0',
                            nonce: '<?php echo esc_js(wp_create_nonce('fukurochat_sync_toggle')); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                // If enabling, start initial sync
                                if (isEnabled) {
                                    $saving.html('🔄 Starting initial sync...');
                                    performInitialSync($saving, $toggle);
                                } else {
                                    $saving.css({
                                        'background': '#d4edda',
                                        'border-color': '#c3e6cb',
                                        'color': '#155724'
                                    }).html('✓ Product sync disabled');
                                    
                                    setTimeout(function() {
                                        $saving.fadeOut(300, function() {
                                            $(this).remove();
                                        });
                                    }, 3000);
                                }
                            } else {
                                $saving.css({
                                    'background': '#f8d7da',
                                    'border-color': '#f5c6cb',
                                    'color': '#721c24'
                                }).html('✗ Failed to update: ' + response.data);
                                $toggle.prop('checked', !isEnabled);
                            }
                        },
                        error: function() {
                            $saving.css({
                                'background': '#f8d7da',
                                'border-color': '#f5c6cb',
                                'color': '#721c24'
                            }).html('✗ Failed to save. Please try again.');
                            $toggle.prop('checked', !isEnabled);
                        }
                    });
                });
                
                // Function to perform initial sync with progress
                function performInitialSync($message, $toggle) {
                    var synced = 0;
                    var failed = 0;
                    var offset = 0;
                    var batchSize = 10;
                    var total = 0;
                    
                    function syncBatch() {
                        $.ajax({
                            url: ajaxurl,
                            method: 'POST',
                            data: {
                                action: 'fukurochat_initial_sync',
                                offset: offset,
                                batch_size: batchSize,
                                nonce: '<?php echo esc_js(wp_create_nonce('fukurochat_initial_sync')); ?>'
                            },
                            success: function(response) {
                                if (response.success) {
                                    var data = response.data;
                                    synced += data.synced;
                                    failed += data.failed;
                                    total = data.total;
                                    offset += batchSize;
                                    
                                    var progress = Math.min(100, Math.round((offset / total) * 100));
                                    var current = Math.min(offset, total);
                                    
                                    if (data.has_more) {
                                        // Update progress message
                                        $message.html('🔄 Synchronizing... ' + current + '/' + total + ' products (' + progress + '%)');
                                        // Continue with next batch
                                        syncBatch();
                                    } else {
                                        // Sync complete
                                        $message.css({
                                            'background': '#d4edda',
                                            'border-color': '#c3e6cb',
                                            'color': '#155724'
                                        }).html('✓ Sync enabled! ' + synced + ' product(s) synchronized' + (failed > 0 ? ', ' + failed + ' failed' : ''));
                                        
                                        setTimeout(function() {
                                            $message.fadeOut(300, function() {
                                                $(this).remove();
                                            });
                                        }, 5000);
                                    }
                                } else {
                                    $message.css({
                                        'background': '#fff3cd',
                                        'border-color': '#ffc107',
                                        'color': '#856404'
                                    }).html('⚠ Sync enabled but initial sync failed: ' + response.data);
                                }
                            },
                            error: function() {
                                $message.css({
                                    'background': '#fff3cd',
                                    'border-color': '#ffc107',
                                    'color': '#856404'
                                }).html('⚠ Sync enabled but initial sync encountered an error');
                            }
                        });
                    }
                    
                    // Start syncing
                    $message.html('🔄 Counting products...');
                    syncBatch();
                }
            });
            </script>
        </div>
        <?php
    }

    public function inject_fukurochat_widget() {
        // Check if widget is enabled in Customizer (defaults to true)
        $widget_enabled = get_theme_mod('fukurochat_widget_enabled', true);
        
        if (!$widget_enabled) {
            return;
        }

        // Get the saved UUID
        $uuid = get_option('fukurochat_assistant_uuid', '');

        // Only inject if UUID is configured
        if (empty($uuid)) {
            return;
        }

        // Get the optional settings from Customizer (with fallback to plugin settings)
        $bubble_text = get_theme_mod('fukurochat_bubble_text', get_option('fukurochat_bubble_text', ''));
        $target_id = get_theme_mod('fukurochat_target_id', get_option('fukurochat_target_id', ''));
        $position = get_theme_mod('fukurochat_position', get_option('fukurochat_position', 'bottom-right'));
        $offset_h = get_theme_mod('fukurochat_offset_horizontal', 24);
        $offset_v = get_theme_mod('fukurochat_offset_vertical', 24);

        // Build the widget script manually (wp_enqueue_script doesn't support external async scripts well)
        $widget_config = wp_json_encode(array(
            'TARGET' => $target_id,
            'TEXT' => $bubble_text,
            'POSITION' => $position,
            'OFFSET_H' => intval($offset_h),
            'OFFSET_V' => intval($offset_v),
        ), JSON_UNESCAPED_SLASHES);
        
        // Output inline script directly (we're already in wp_footer hook)
        echo "\n<script>\n";
        echo "(function(w,d,s,id){\n";
        echo "if(d.getElementById(id))return;\n";
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON already escaped by wp_json_encode
        echo "w.FukuroChat=" . $widget_config . ";\n";
        echo "var js=d.createElement(s),fjs=d.getElementsByTagName(s)[0];\n";
        echo "js.id=id;\n";
        echo "js.async=true;\n";
        echo "js.src='https://app.fukurochat.com/embed/widget.min.js?id=" . esc_js($uuid) . "';\n";
        echo "fjs.parentNode.insertBefore(js,fjs);\n";
        echo "})(window,document,'script','fukurochat-sdk');\n";
        echo "</script>\n";
    }
    
    // Validate product has required fields for sync
    private function validate_product_for_sync($product) {
        $title = $product->get_name();
        $description = $product->get_description();

        if (empty($title)) {
            set_transient('fukurochat_last_sync_result', array(
                'success' => false,
                'message' => 'Product must have a title to be synced'
            ), 30);
            return false;
        }

        if (empty($description)) {
            set_transient('fukurochat_last_sync_result', array(
                'success' => false,
                'message' => 'Product must have a description to be synced'
            ), 30);
            return false;
        }

        // SKU is optional - many products don't have one
        return true;
    }

    // Sync product when created or updated
    public function sync_product_create($product_id) {
        // This hook is not reliable, just mark the product for the update hook
        update_post_meta($product_id, '_fukurochat_is_new', '1');
    }

    // Sync product when updated (this is the main sync point)
    public function sync_product_update($product_id) {
        // Debug logging
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: sync_product_update called for product ID: $product_id");
        
        $sync_enabled = get_option('fukurochat_sync_enabled', '0');
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: Sync enabled: " . ($sync_enabled === '1' ? 'YES' : 'NO'));
        
        if ($sync_enabled !== '1') {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Sync is disabled, exiting");
            return;
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Product not found for ID: $product_id");
            return;
        }

        // Only sync published products - draft/pending products are handled by transition_post_status
        $post_status = get_post_status($product_id);
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: Product status: $post_status");
        
        if ($post_status !== 'publish') {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Product not published, exiting");
            return;
        }

        // Validate product has required fields
        if (!$this->validate_product_for_sync($product)) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Product validation failed");
            return;
        }
        
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: Product passed all checks, proceeding with sync");

        // Check if product has been synced to FukuroChat before
        $has_been_synced = get_post_meta($product_id, '_fukurochat_synced', true);
        $is_new_marker = get_post_meta($product_id, '_fukurochat_is_new', true);
        
        // If it has the "new" marker OR has never been synced, use add_article
        if ($is_new_marker === '1' || empty($has_been_synced)) {
            // This is a new product, use add_article
            delete_post_meta($product_id, '_fukurochat_is_new');
            update_post_meta($product_id, '_fukurochat_synced', '1');
            $this->call_fukurochat_api('add_article', $product);
        } else {
            // This is an existing product that has been synced before, use edit_article
            $this->call_fukurochat_api('edit_article', $product);
        }
    }

    // Sync product when moved to trash
    public function sync_product_trash($post_id) {
        if (get_option('fukurochat_sync_enabled', '0') !== '1') {
            return;
        }

        // Check if it's a product
        if (get_post_type($post_id) !== 'product') {
            return;
        }

        // Check if this product was ever synced with FukuroChat
        $has_been_synced = get_post_meta($post_id, '_fukurochat_synced', true);
        if (empty($has_been_synced)) {
            // Product was never synced, no need to delete from API
            return;
        }

        $product = wc_get_product($post_id);
        if (!$product) {
            return;
        }

        $this->call_fukurochat_api('delete_article', $product);
        
        // Clean up meta fields
        delete_post_meta($post_id, '_fukurochat_synced');
        delete_post_meta($post_id, '_fukurochat_is_new');
    }

    // Sync product when permanently deleted
    public function sync_product_delete($post_id) {
        if (get_option('fukurochat_sync_enabled', '0') !== '1') {
            return;
        }

        // Check if it's a product
        if (get_post_type($post_id) !== 'product') {
            return;
        }

        // Check if this product was ever synced with FukuroChat
        $has_been_synced = get_post_meta($post_id, '_fukurochat_synced', true);
        if (empty($has_been_synced)) {
            // Product was never synced, no need to delete from API
            return;
        }

        $product = wc_get_product($post_id);
        if (!$product) {
            return;
        }

        $this->call_fukurochat_api('delete_article', $product);
        
        // Clean up meta fields
        delete_post_meta($post_id, '_fukurochat_synced');
        delete_post_meta($post_id, '_fukurochat_is_new');
    }

    // Handle product status changes (draft, publish, etc.)
    public function sync_product_status_change($new_status, $old_status, $post) {
        if (get_option('fukurochat_sync_enabled', '0') !== '1') {
            return;
        }

        // Only handle product post type
        if ($post->post_type !== 'product') {
            return;
        }

        $product_id = $post->ID;
        $has_been_synced = get_post_meta($product_id, '_fukurochat_synced', true);

        // Case 1: Product is being published (from draft, pending, etc.)
        if ($new_status === 'publish' && $old_status !== 'publish') {
            $product = wc_get_product($product_id);
            if ($product && $this->validate_product_for_sync($product)) {
                if ($has_been_synced) {
                    // Product was already in FukuroChat before being unpublished
                    // Use edit_article to update it (not add a new one)
                    $this->call_fukurochat_api('edit_article', $product);
                } else {
                    // Product was never synced OR was deleted from FukuroChat when unpublished
                    // Add it as a new article
                    $this->call_fukurochat_api('add_article', $product);
                    update_post_meta($product_id, '_fukurochat_synced', '1');
                }
            }
        }
        
        // Case 2: Product is being unpublished (to draft, pending, private, etc.)
        if ($old_status === 'publish' && $new_status !== 'publish') {
            // Remove from FukuroChat if it was synced
            if ($has_been_synced) {
                $product = wc_get_product($product_id);
                if ($product) {
                    $this->call_fukurochat_api('delete_article', $product);
                    // Remove sync flag since product no longer exists in FukuroChat
                    // When republished, it will be added as a new article
                    delete_post_meta($product_id, '_fukurochat_synced');
                }
            }
        }
    }

    // Clean text before sending to API
    private function fukurochat_cleanText($text) {
        // Esta función limpia el texto eliminando saltos de línea extra y espacios no separables (&nbsp;)
        $text = str_replace('&nbsp;', ' ', $text); // reemplazar &nbsp; por espacio
        $text = preg_replace('/\r\n|\r|\n/', "\n", $text); // normalizar saltos de línea
        $text = preg_replace('/\n{2,}/', "\n", $text); // eliminar saltos de línea consecutivos
        $text = trim($text); // quitar espacios al inicio y final
        return $text;
    }

    // Build product data array for API
    private function build_product_data($product) {
        $data = array(
            'productId' => $product->get_id(),
            'articleTitle' => $product->get_name(),
            'articleUrl' => $product->get_permalink()
        );
        
        // SKU
        $sku = $product->get_sku();
        $data['articleSku'] = !empty($sku) ? $sku : '';
        
        // Prices
        $data['articlePriceRegular'] = $product->get_regular_price() ? floatval($product->get_regular_price()) : 0;
        $data['articlePriceSale'] = $product->get_sale_price() ? floatval($product->get_sale_price()) : 0;
        
        // Sale dates
        $sale_from = $product->get_date_on_sale_from();
        $sale_to = $product->get_date_on_sale_to();
        $data['articleSaleFrom'] = $sale_from ? $sale_from->format('Y-m-d') : '';
        $data['articleSaleTo'] = $sale_to ? $sale_to->format('Y-m-d') : '';
        
        // Stock status (instock, outofstock, onbackorder)
        $data['articleStockStatus'] = $product->get_stock_status();
        
        // Build enriched product text with all details (WITHOUT SKU and prices)
        $enriched_text = '';
        
        // Title
        $enriched_text .= $product->get_name() . "\n";
        
        // Short description (excerpt)
        $short_desc = wp_strip_all_tags($product->get_short_description());
        if (!empty($short_desc)) {
            $enriched_text .= $this->fukurochat_cleanText($short_desc) . "\n";
        }
        
        // Full description
        $enriched_text .= "\n";
        $full_desc = wp_strip_all_tags($product->get_description());
        if (!empty($full_desc)) {
            $enriched_text .= $this->fukurochat_cleanText($full_desc) . "\n";
        }
        
        // Attributes
        $attributes = $product->get_attributes();
        if (!empty($attributes)) {
            $enriched_text .= "\nAttributes:\n";
            foreach ($attributes as $attribute) {
                if (is_a($attribute, 'WC_Product_Attribute')) {
                    $name = wc_attribute_label($attribute->get_name());
                    $values = $attribute->get_options();
                    if (is_array($values)) {
                        $term_names = array();
                        foreach ($values as $term_id) {
                            $term = get_term($term_id);
                            if ($term && !is_wp_error($term)) {
                                $term_names[] = $term->name;
                            }
                        }
                        if (!empty($term_names)) {
                            $enriched_text .= "  - " . $name . ": " . implode(', ', $term_names) . "\n";
                        }
                    } else {
                        $enriched_text .= "  - " . $name . ": " . $values . "\n";
                    }
                }
            }
        }
        
        // Categories
        $category_ids = $product->get_category_ids();
        if (!empty($category_ids)) {
            $category_names = array();
            foreach ($category_ids as $cat_id) {
                $term = get_term($cat_id, 'product_cat');
                if ($term && !is_wp_error($term)) {
                    $category_names[] = $term->name;
                }
            }
            if (!empty($category_names)) {
                $enriched_text .= "\nCategories: " . implode(', ', $category_names) . "\n";
            }
        }
        
        // Upsells (ventas dirigidas)
        $upsell_ids = $product->get_upsell_ids();
        if (!empty($upsell_ids)) {
            $upsell_names = array();
            foreach ($upsell_ids as $upsell_id) {
                $upsell_product = wc_get_product($upsell_id);
                if ($upsell_product) {
                    $upsell_names[] = $upsell_product->get_name();
                }
            }
            if (!empty($upsell_names)) {
                $enriched_text .= "\nRecommended products:\n";
                foreach ($upsell_names as $name) {
                    $enriched_text .= "  - " . $name . "\n";
                }
            }
        }
        
        // Cross-sells (ventas cruzadas)
        $crosssell_ids = $product->get_cross_sell_ids();
        if (!empty($crosssell_ids)) {
            $crosssell_names = array();
            foreach ($crosssell_ids as $crosssell_id) {
                $crosssell_product = wc_get_product($crosssell_id);
                if ($crosssell_product) {
                    $crosssell_names[] = $crosssell_product->get_name();
                }
            }
            if (!empty($crosssell_names)) {
                $enriched_text .= "\nFrequently bought together:\n";
                foreach ($crosssell_names as $name) {
                    $enriched_text .= "  - " . $name . "\n";
                }
            }
        }
        
        // Alternative products (FukuroChat custom field)
        $alternative_products = get_post_meta($product->get_id(), '_fukurochat_alternative_products', true);
        if (!empty($alternative_products)) {
            $enriched_text .= "\nThis product is an alternative to: " . $alternative_products . "\n";
        }
        
        $data['articleText'] = $enriched_text;
        
        return $data;
    }

    /**
     * Send multiple products to FukuroChat API in a single batch request
     * 
     * @param array $products Array of WC_Product objects
     * @return array Results with synced/failed counts
     */
    private function call_fukurochat_api_batch($products) {
        $uuid = get_option('fukurochat_assistant_uuid', '');
        $secret = get_option('fukurochat_api_secret', '');

        if (empty($uuid) || empty($secret)) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Missing UUID or Secret, cannot batch sync");
            return array('synced' => 0, 'failed' => count($products), 'results' => array());
        }

        $api_url = 'https://app.fukurochat.com/api/index.php';
        
        // Build products array for batch request
        $products_data = array();
        $product_ids = array();
        
        foreach ($products as $product) {
            $product_data = $this->build_product_data($product);
            $products_data[] = $product_data;
            $product_ids[] = $product->get_id();
        }
        
        $json_data = array(
            'assistantUUID' => $uuid,
            'secret' => $secret,
            'action' => 'add_article',
            'source' => 'woocommerce',
            'products' => $products_data
        );
        
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: Sending batch request with " . count($products_data) . " products");
        
        $response = wp_remote_post($api_url, array(
            'method'      => 'POST',
            'timeout'     => 60, // Longer timeout for batch
            'headers'     => array('Content-Type' => 'application/json'),
            'body'        => wp_json_encode($json_data),
        ));

        $synced = 0;
        $failed = 0;
        $results = array();

        if (is_wp_error($response)) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Batch API request failed - WP Error: " . $response->get_error_message());
            return array('synced' => 0, 'failed' => count($products), 'results' => array());
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: Batch API response - " . substr($body, 0, 500));
        
        if (isset($result['data']['results']) && is_array($result['data']['results'])) {
            foreach ($result['data']['results'] as $index => $item_result) {
                $product_id = isset($product_ids[$index]) ? $product_ids[$index] : 0;
                
                if (isset($item_result['status']) && $item_result['status']) {
                    $synced++;
                    if ($product_id) {
                        update_post_meta($product_id, '_fukurochat_synced', '1');
                        delete_post_meta($product_id, '_fukurochat_is_new');
                    }
                } else {
                    $failed++;
                }
                
                $results[] = array(
                    'product_id' => $product_id,
                    'status' => $item_result['status'] ?? false,
                    'error' => $item_result['error'] ?? ''
                );
            }
        } else {
            // Fallback: if no detailed results, check overall status
            if (isset($result['status']) && $result['status']) {
                $synced = count($products);
                foreach ($product_ids as $product_id) {
                    update_post_meta($product_id, '_fukurochat_synced', '1');
                    delete_post_meta($product_id, '_fukurochat_is_new');
                }
            } else {
                $failed = count($products);
            }
        }
        
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: Batch sync results - Synced: $synced, Failed: $failed");
        
        return array('synced' => $synced, 'failed' => $failed, 'results' => $results);
    }

    // Make API call to FukuroChat
    private function call_fukurochat_api($action, $product) {
        $uuid = get_option('fukurochat_assistant_uuid', '');
        $secret = get_option('fukurochat_api_secret', '');

        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: call_fukurochat_api - Action: $action, Product ID: " . $product->get_id());
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: UUID configured: " . (!empty($uuid) ? 'YES' : 'NO'));
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log("FukuroChat: Secret configured: " . (!empty($secret) ? 'YES' : 'NO'));

        if (empty($uuid) || empty($secret)) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Missing UUID or Secret, cannot sync");
            return false;
        }

        $api_url = 'https://app.fukurochat.com/api/index.php';

        if ($action === 'delete_article') {
            // Delete uses form data (not JSON)
            $data = array(
                'assistantUUID' => $uuid,
                'secret' => $secret,
                'action' => $action,
                'source' => 'woocommerce',
                'productId' => $product->get_id()
            );
            
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Sending DELETE request to API...");
            $response = wp_remote_post($api_url, array(
                'method'      => 'POST',
                'timeout'     => 15,
                'body'        => $data,
            ));
        } else {
            // Add/edit uses JSON with products array
            $product_data = $this->build_product_data($product);
            
            $json_data = array(
                'assistantUUID' => $uuid,
                'secret' => $secret,
                'action' => 'add_article',
                'source' => 'woocommerce',
                'products' => array($product_data)
            );
            
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Sending product - Title: " . $product_data['articleTitle']);
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Text length: " . strlen($product_data['articleText']) . " chars");
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: Sending POST request to API (JSON)...");
            
            $response = wp_remote_post($api_url, array(
                'method'      => 'POST',
                'timeout'     => 15,
                'headers'     => array('Content-Type' => 'application/json'),
                'body'        => json_encode($json_data),
            ));
        }

        // Handle response and save result for admin notice
        if (is_wp_error($response)) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: API request failed - WP Error: " . $response->get_error_message());
            set_transient('fukurochat_last_sync_result', array(
                'success' => false,
                'message' => 'Connection error: ' . $response->get_error_message()
            ), 30);
            return false;
        } else {
            $body = wp_remote_retrieve_body($response);
            $result = json_decode($body, true);
            
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: API response body: " . substr($body, 0, 500));
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("FukuroChat: API response parsed - Status: " . (isset($result['status']) ? ($result['status'] ? 'true' : 'false') : 'not set'));
            
            if (isset($result['status']) && $result['status']) {
                // Success - show user-friendly message based on action
                $action_text = '';
                switch ($action) {
                    case 'add_article':
                        $action_text = 'Product added to FukuroChat successfully!';
                        break;
                    case 'edit_article':
                        $action_text = 'Product updated in FukuroChat successfully!';
                        break;
                    case 'delete_article':
                        $action_text = 'Product removed from FukuroChat successfully!';
                        break;
                    default:
                        $action_text = 'Product synchronized successfully!';
                }
                
                set_transient('fukurochat_last_sync_result', array(
                    'success' => true,
                    'message' => $action_text
                ), 30);
                return true;
            } else {
                // API returned error
                $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';
                
                // Special case: if edit_article fails because product doesn't exist in FukuroChat,
                // try to add it instead
                if ($action === 'edit_article' && strpos($error_msg, 'productId') !== false) {
                    // Product doesn't exist in FukuroChat, try adding it
                    $product_data = $this->build_product_data($product);
                    
                    $retry_json_data = array(
                        'assistantUUID' => $uuid,
                        'secret' => $secret,
                        'action' => 'add_article',
                        'source' => 'woocommerce',
                        'products' => array($product_data)
                    );
                    
                    $retry_response = wp_remote_post($api_url, array(
                        'method'      => 'POST',
                        'timeout'     => 15,
                        'headers'     => array('Content-Type' => 'application/json'),
                        'body'        => json_encode($retry_json_data),
                    ));
                    
                    if (!is_wp_error($retry_response)) {
                        $retry_body = wp_remote_retrieve_body($retry_response);
                        $retry_result = json_decode($retry_body, true);
                        
                        if (isset($retry_result['status']) && $retry_result['status']) {
                            // Successfully added the product
                            set_transient('fukurochat_last_sync_result', array(
                                'success' => true,
                                'message' => 'Product was missing in FukuroChat and has been added successfully!'
                            ), 30);
                            return true; // Exit early after successful retry
                        }
                    }
                }
                
                // Show the original error if retry didn't happen or also failed
                $debug_msg = $error_msg . ' | Action: ' . $action . ' | Product ID: ' . $product->get_id();
                set_transient('fukurochat_last_sync_result', array(
                    'success' => false,
                    'message' => $debug_msg
                ), 30);
                return false;
            }
        }
        return false;
    }

    /**
     * AJAX handler to track add-to-cart events
     */
    /**
     * AJAX handler to exchange connection token for credentials
     */
    public function ajax_exchange_token() {
        // Verify nonce
        check_ajax_referer('fukurochat_connect', 'nonce');
        
        // Only admins can connect
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
        $token = isset($_POST['token']) ? sanitize_text_field(wp_unslash($_POST['token'])) : '';
        
        if (empty($token)) {
            wp_send_json_error('Token is required');
            return;
        }
        
        // Exchange token with FukuroChat API
        $api_url = 'https://app.fukurochat.com/webservices/connector/api.php';
        
        $response = wp_remote_post($api_url, array(
            'timeout' => 30,
            'body' => array(
                'action' => 'exchange_connection_token',
                'token' => $token
            )
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error('Failed to connect to FukuroChat API: ' . $response->get_error_message());
            return;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!isset($data['status']) || $data['status'] === false) {
            $error = !empty($data['error']) ? $data['error'] : 'Invalid or expired token';
            wp_send_json_error($error);
            return;
        }
        
        // Extract credentials from response
        $uuid = !empty($data['data']['assistantUUID']) ? $data['data']['assistantUUID'] : '';
        $secret = !empty($data['data']['apiSecret']) ? $data['data']['apiSecret'] : '';
        $name = !empty($data['data']['assistantName']) ? $data['data']['assistantName'] : '';
        
        if (empty($uuid) || empty($secret)) {
            wp_send_json_error('Invalid credentials received from API');
            return;
        }
        
        // Save credentials
        update_option('fukurochat_assistant_uuid', $uuid);
        update_option('fukurochat_api_secret', $secret);
        update_option('fukurochat_assistant_name', $name);
        
        // Mark credentials as valid
        update_option('fukurochat_credentials_valid', array(
            'valid' => true,
            'checked_at' => current_time('mysql'),
            'connected_via' => 'oauth'
        ));
        
        wp_send_json_success(array(
            'message' => 'Successfully connected to ' . $name,
            'assistant_name' => $name,
            'assistant_uuid' => $uuid
        ));
    }
    
    public function ajax_toggle_sync() {
        error_log("FukuroChat: ajax_toggle_sync called");
        
        // Verify nonce
        check_ajax_referer('fukurochat_sync_toggle', 'nonce');
        
        // Only admins can toggle sync
        if (!current_user_can('manage_options')) {
            error_log("FukuroChat: User not authorized to toggle sync");
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
        $enabled = isset($_POST['enabled']) ? sanitize_text_field(wp_unslash($_POST['enabled'])) : '0';
        // Keep as string '1' or '0' - WordPress handles these better than booleans
        $enabled = ($enabled === '1') ? '1' : '0';
        
        error_log("FukuroChat: Attempting to save sync_enabled = " . $enabled);
        
        // Use update_option - autoload=true ensures value persists correctly across page loads
        $result = update_option('fukurochat_sync_enabled', $enabled, true);
        
        error_log("FukuroChat: update_option result: " . ($result ? 'SUCCESS' : 'NO_CHANGE'));
        
        // Verify it was saved
        $saved_value = get_option('fukurochat_sync_enabled', 'NOT_SET');
        error_log("FukuroChat: Value after save: " . var_export($saved_value, true));
        
        wp_send_json_success(array(
            'message' => $enabled === '1' ? 'Sync enabled' : 'Sync disabled',
            'enabled' => $enabled === '1',
            'saved_value' => $saved_value
        ));
    }
    
    public function ajax_initial_sync() {
        error_log("FukuroChat: ajax_initial_sync called");
        
        // Verify nonce
        check_ajax_referer('fukurochat_initial_sync', 'nonce');
        
        // Only admins can trigger sync
        if (!current_user_can('manage_options')) {
            error_log("FukuroChat: User not authorized to trigger initial sync");
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // Get credentials
        $uuid = get_option('fukurochat_assistant_uuid', '');
        $secret = get_option('fukurochat_api_secret', '');
        
        if (empty($uuid) || empty($secret)) {
            wp_send_json_error('Missing assistant UUID or API secret');
            return;
        }
        
        // Get pagination parameters
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
        $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : 10;
        
        // First get total count
        $count_args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        );
        // phpcs:ignore WordPress.WP.PostsPerPage.posts_per_page_posts_per_page -- Need to count all products
        $all_product_ids = get_posts($count_args);
        $total = count($all_product_ids);
        
        if ($total === 0) {
            wp_send_json_success(array(
                'synced' => 0,
                'failed' => 0,
                'total' => 0,
                'has_more' => false,
                'message' => 'No products to sync'
            ));
            return;
        }
        
        // Get batch of products
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => $batch_size,
            'offset' => $offset,
            'fields' => 'ids',
            'orderby' => 'ID',
            'order' => 'ASC'
        );
        $product_ids = get_posts($args);
        
        // Collect valid products for batch sync
        $products_to_sync = array();
        $failed = 0;
        
        foreach ($product_ids as $product_id) {
            $product = wc_get_product($product_id);
            
            if (!$product) {
                $failed++;
                continue;
            }
            
            // Validate product has required fields
            if (!$this->validate_product_for_sync($product)) {
                $failed++;
                continue;
            }
            
            $products_to_sync[] = $product;
        }
        
        // Batch sync all valid products in a single API call
        $synced = 0;
        if (!empty($products_to_sync)) {
            $batch_result = $this->call_fukurochat_api_batch($products_to_sync);
            $synced = $batch_result['synced'];
            $failed += $batch_result['failed'];
        }
        
        // Check if there are more products
        $has_more = ($offset + $batch_size) < $total;
        
        error_log("FukuroChat: Batch sync - Offset: $offset, Batch: $batch_size, Synced: $synced, Failed: $failed, Has more: " . ($has_more ? 'yes' : 'no'));
        
        wp_send_json_success(array(
            'synced' => $synced,
            'failed' => $failed,
            'total' => $total,
            'has_more' => $has_more
        ));
    }

    public function ajax_track_add_to_cart() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Public endpoint for tracking
        $sku = isset($_POST['sku']) ? sanitize_text_field(wp_unslash($_POST['sku'])) : '';
        
        if (empty($sku)) {
            wp_send_json_error('SKU is required');
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'fukurochat_add_to_cart';
        
        // Get or create session ID
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $session_id = session_id();
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->insert(
            $table_name,
            array(
                'product_sku' => $sku,
                'session_id' => $session_id,
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%s')
        );

        if ($result) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('FukuroChat: Tracked add-to-cart for SKU: ' . $sku);
            wp_send_json_success('Tracked');
        } else {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('FukuroChat: Failed to track add-to-cart - Error: ' . $wpdb->last_error);
            wp_send_json_error('Database error');
        }
    }

    /**
     * Inject JavaScript to track when products are added via chatbot
     */
    public function inject_conversion_tracking_script() {
        ?>
        <script>
        // Listen for add-to-cart events from FukuroChat widget
        window.addEventListener('message', function(event) {
            if (event.data && event.data.type === 'fukurochat_product_added_to_cart') {
                // Store in session that this session has chatbot interactions
                if (typeof(Storage) !== "undefined") {
                    var chatbotProducts = sessionStorage.getItem('fukurochat_products') || '[]';
                    var products = JSON.parse(chatbotProducts);
                    products.push({
                        sku: event.data.sku,
                        timestamp: Date.now()
                    });
                    sessionStorage.setItem('fukurochat_products', JSON.stringify(products));
                    sessionStorage.setItem('fukurochat_assisted', '1');
                }

                // Send AJAX request to track this add-to-cart event in database
                jQuery.ajax({
                    url: '<?php echo esc_url(admin_url('admin-ajax.php')); ?>',
                    type: 'POST',
                    data: {
                        action: 'fukurochat_track_add_to_cart',
                        sku: event.data.sku
                    },
                    success: function(response) {
                        console.log('FukuroChat: Add-to-cart tracked', response);
                    },
                    error: function(error) {
                        console.error('FukuroChat: Failed to track add-to-cart', error);
                    }
                });
            }
        });
        </script>
        <?php
        
        // On checkout page, inject script to send sessionStorage data during checkout
        if (is_checkout()) {
            ?>
            <script>
            jQuery(document).ready(function($) {
                // When checkout form is submitted, attach chatbot data as hidden fields
                $('form.checkout').on('checkout_place_order', function() {
                    if (typeof(Storage) !== "undefined") {
                        var chatbotAssisted = sessionStorage.getItem('fukurochat_assisted');
                        var chatbotProducts = sessionStorage.getItem('fukurochat_products');
                        
                        if (chatbotAssisted) {
                            // Remove existing hidden fields if any
                            $('input[name="fukurochat_assisted"]').remove();
                            $('input[name="fukurochat_products"]').remove();
                            
                            // Add hidden fields to form
                            $(this).append('<input type="hidden" name="fukurochat_assisted" value="1">');
                            if (chatbotProducts) {
                                $(this).append('<input type="hidden" name="fukurochat_products" value=\'' + chatbotProducts + '\'>');
                            }
                        }
                    }
                    return true; // Allow form submission to continue
                });
            });
            </script>
            <?php
        }
    }

    /**
     * Save chatbot meta data to order during checkout (traditional checkout)
     */
    public function save_chatbot_meta_to_order($order, $data) {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce handles nonce verification
        if (isset($_POST['fukurochat_assisted']) && $_POST['fukurochat_assisted'] === '1') {
            $order->update_meta_data('_fukurochat_assisted', 1);
            
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            if (isset($_POST['fukurochat_products'])) {
                $order->update_meta_data('_fukurochat_products', sanitize_text_field(wp_unslash($_POST['fukurochat_products'])));
            }
        }
    }

    /**
     * Save chatbot meta data to order (Store API / Block checkout)
     */
    public function save_chatbot_meta_to_order_api($order) {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce Store API handles security
        if (isset($_POST['fukurochat_assisted']) && $_POST['fukurochat_assisted'] === '1') {
            $order->update_meta_data('_fukurochat_assisted', 1);
            
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            if (isset($_POST['fukurochat_products'])) {
                $order->update_meta_data('_fukurochat_products', sanitize_text_field(wp_unslash($_POST['fukurochat_products'])));
            }
            $order->save();
        }
    }

    /**
     * Track order conversion when order is completed
     */
    public function track_order_conversion($order_id) {
        if (!$order_id) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('FukuroChat: track_order_conversion called without order_id');
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('FukuroChat: Order not found for ID: ' . $order_id);
            return;
        }

        // Check if this order was already tracked
        $already_tracked = $order->get_meta('_fukurochat_tracked');
        if ($already_tracked) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('FukuroChat: Order ' . $order_id . ' already tracked, skipping');
            return;
        }

        // Get chatbot products from order meta (set during checkout)
        $chatbot_assisted = $order->get_meta('_fukurochat_assisted');
        $chatbot_products = $order->get_meta('_fukurochat_products');

        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log('FukuroChat: Order ' . $order_id . ' - Assisted: ' . ($chatbot_assisted ? 'YES' : 'NO'));

        // If not set via checkout hook, try to get from POST data (fallback)
        if (!$chatbot_assisted) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            if (isset($_POST['fukurochat_assisted']) && $_POST['fukurochat_assisted'] === '1') {
                $chatbot_assisted = 1;
                $order->update_meta_data('_fukurochat_assisted', 1);
                
                // phpcs:ignore WordPress.Security.NonceVerification.Missing
                if (isset($_POST['fukurochat_products'])) {
                    $chatbot_products = sanitize_text_field(wp_unslash($_POST['fukurochat_products']));
                    $order->update_meta_data('_fukurochat_products', $chatbot_products);
                }
                $order->save();
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('FukuroChat: Saved meta from POST for order ' . $order_id);
            }
        }

        if ($chatbot_assisted) {
            // Mark order as chatbot-assisted
            $order->update_meta_data('_fukurochat_conversion', 1);
            $order->update_meta_data('_fukurochat_tracked', 1);
            $order->save();

            // Store in global conversion table
            global $wpdb;
            $table_name = $wpdb->prefix . 'fukurochat_conversions';
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $result = $wpdb->insert(
                $table_name,
                array(
                    'order_id' => $order_id,
                    'order_total' => $order->get_total(),
                    'chatbot_products' => $chatbot_products ? $chatbot_products : '',
                    'created_at' => current_time('mysql')
                ),
                array('%d', '%f', '%s', '%s')
            );

            if ($result) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('FukuroChat: Successfully tracked conversion for order ' . $order_id);
            } else {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('FukuroChat: Failed to insert conversion for order ' . $order_id . ' - Error: ' . $wpdb->last_error);
            }
        } else {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('FukuroChat: Order ' . $order_id . ' not assisted by chatbot');
        }
    }

    /**
     * Render Analytics page
     */
    public function render_analytics_page() {
        global $wpdb;
        $conversions_table = $wpdb->prefix . 'fukurochat_conversions';
        $add_to_cart_table = $wpdb->prefix . 'fukurochat_add_to_cart';
        
        // Get date range from query params
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just for filtering display
        $days = isset($_GET['days']) ? intval($_GET['days']) : 30;
        $date_from = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        // Products added to cart via chatbot
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $products_added = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$add_to_cart_table} WHERE created_at >= %s",
                $date_from
            )
        );
        
        // Total orders in period
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $total_orders = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}wc_orders WHERE date_created_gmt >= %s",
                $date_from
            )
        );
        
        // Chatbot-assisted orders (conversions)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $chatbot_orders = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$conversions_table} WHERE created_at >= %s",
                $date_from
            )
        );
        
        // Total revenue from chatbot
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $chatbot_revenue = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT SUM(order_total) FROM {$conversions_table} WHERE created_at >= %s",
                $date_from
            )
        );
        
        $conversion_rate = $total_orders > 0 ? round(($chatbot_orders / $total_orders) * 100, 1) : 0;
        $add_to_cart_conversion = $products_added > 0 ? round(($chatbot_orders / $products_added) * 100, 1) : 0;
        
        ?>
        <div class="wrap">
            <h1>FukuroChat Analytics</h1>
            
            <div class="fukuro-analytics-filters" style="margin: 20px 0;">
                <form method="get" action="">
                    <input type="hidden" name="page" value="fukurochat-analytics">
                    <select name="days" onchange="this.form.submit()">
                        <option value="7" <?php selected($days, 7); ?>>Last 7 days</option>
                        <option value="30" <?php selected($days, 30); ?>>Last 30 days</option>
                        <option value="90" <?php selected($days, 90); ?>>Last 90 days</option>
                    </select>
                </form>
            </div>

            <div class="fukuro-analytics-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 20px 0;">
                
                <div class="fukuro-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 10px 0; color: #666; font-size: 14px;">Products Added to Cart</h3>
                    <p style="margin: 0; font-size: 32px; font-weight: bold; color: #9C27B0;"><?php echo esc_html(number_format($products_added)); ?></p>
                </div>

                <div class="fukuro-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 10px 0; color: #666; font-size: 14px;">Chatbot-Assisted Orders</h3>
                    <p style="margin: 0; font-size: 32px; font-weight: bold; color: #2196F3;"><?php echo esc_html(number_format($chatbot_orders)); ?></p>
                </div>

                <div class="fukuro-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 10px 0; color: #666; font-size: 14px;">Funnel Conversion</h3>
                    <p style="margin: 0; font-size: 32px; font-weight: bold; color: #4CAF50;"><?php echo esc_html($add_to_cart_conversion); ?>%</p>
                    <p style="margin: 5px 0 0 0; font-size: 12px; color: #999;">Cart → Order</p>
                </div>

                <div class="fukuro-stat-card" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <h3 style="margin: 0 0 10px 0; color: #666; font-size: 14px;">Revenue Generated</h3>
                    <p style="margin: 0; font-size: 32px; font-weight: bold; color: #FF9800;"><?php echo wp_kses_post(wc_price($chatbot_revenue ? $chatbot_revenue : 0)); ?></p>
                </div>

            </div>

            <div class="fukuro-info" style="background: #e3f2fd; padding: 15px; border-radius: 8px; margin: 20px 0;">
                <p style="margin: 0;">
                    <strong>How it works:</strong> These metrics track orders where customers interacted with the FukuroChat assistant before completing their purchase. 
                    When a customer adds a product to cart via the chatbot's "Add to Cart" button, that order is marked as chatbot-assisted.
                </p>
            </div>
        </div>
        <?php
    }

    public function activate() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        // Create add-to-cart tracking table
        $add_to_cart_table = $wpdb->prefix . 'fukurochat_add_to_cart';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange
        $sql1 = "CREATE TABLE IF NOT EXISTS $add_to_cart_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            product_sku varchar(200) NOT NULL,
            session_id varchar(200),
            created_at datetime NOT NULL,
            PRIMARY KEY  (id),
            KEY session_id (session_id),
            KEY created_at (created_at)
        ) $charset_collate;";

        // Create conversions table (orders completed)
        $conversions_table = $wpdb->prefix . 'fukurochat_conversions';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange
        $sql2 = "CREATE TABLE IF NOT EXISTS $conversions_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            order_id bigint(20) NOT NULL,
            order_total decimal(10,2) NOT NULL,
            chatbot_products text,
            created_at datetime NOT NULL,
            PRIMARY KEY  (id),
            KEY order_id (order_id),
            KEY created_at (created_at)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange
        dbDelta($sql1);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange
        dbDelta($sql2);

        flush_rewrite_rules();
    }

    public function deactivate() {
        flush_rewrite_rules();
    }

    /**
     * Add metabox for alternative products field
     */
    public function add_alternative_products_metabox() {
        add_meta_box(
            'fukurochat_alternative_products',
            'FukuroChat - Alternative Products',
            array($this, 'render_alternative_products_metabox'),
            'product',
            'side',
            'default'
        );
    }

    /**
     * Render the alternative products metabox
     */
    public function render_alternative_products_metabox($post) {
        wp_nonce_field('fukurochat_alternative_products_nonce', 'fukurochat_alternative_products_nonce');
        $value = get_post_meta($post->ID, '_fukurochat_alternative_products', true);
        ?>
        <p>
            <label for="fukurochat_alternative_products" style="display:block; margin-bottom:5px; font-weight:600;">
                <?php esc_html_e('This product is an alternative to:', 'fukurochat-connector'); ?>
            </label>
            <textarea 
                id="fukurochat_alternative_products" 
                name="fukurochat_alternative_products" 
                rows="4" 
                style="width:100%;"
                placeholder="e.g., Product A, Product B, Product C"
            ><?php echo esc_textarea($value); ?></textarea>
        </p>
        <p class="description">
            <?php esc_html_e('Enter a comma-separated list of products this item can replace or substitute.', 'fukurochat-connector'); ?>
        </p>
        <?php
    }

    /**
     * Save the alternative products field
     */
    public function save_alternative_products_field($post_id) {
        // Check if nonce is set
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Checked below
        if (!isset($_POST['fukurochat_alternative_products_nonce'])) {
            return;
        }

        // Verify nonce
        // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        if (!wp_verify_nonce(wp_unslash($_POST['fukurochat_alternative_products_nonce']), 'fukurochat_alternative_products_nonce')) {
            return;
        }

        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Check if this is a product
        if (get_post_type($post_id) !== 'product') {
            return;
        }

        // Save the field
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
        if (isset($_POST['fukurochat_alternative_products'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $value = sanitize_textarea_field(wp_unslash($_POST['fukurochat_alternative_products']));
            update_post_meta($post_id, '_fukurochat_alternative_products', $value);
        } else {
            delete_post_meta($post_id, '_fukurochat_alternative_products');
        }
    }
}

// Initialize the plugin
new Fukurochat_Connector();
