<?php

namespace FrontisInteraction\Models\Triggers;

defined('ABSPATH') || exit;

/**
 * Abstract class for all triggers.
 *
 * This class is the base class for all triggers, such as ClickTrigger, HoverTrigger, etc.
 *
 * It provides a basic structure for triggers.
 *
 * @package FrontisInteraction.Models\Triggers
 * @since 1.1.0
 */
abstract class Trigger implements TriggerInterface
{
  /**
   * The ID of the trigger.
   * 
   * @var string
   * @since 1.1.0
   */
  protected string $id;

  /**
   * The type of the trigger.
   * 
   * @var string
   * @since 1.1.0
   */
  protected string $type;

  /**
   * The action of the trigger.
   * 
   * @var string
   * @since 1.1.0
   */
  private string $action;


  /**
   * Constructor.
   *
   * Creates a new Trigger object with the given data.
   *
   * If no `id` is provided, a unique ID will be generated.
   * If no `type` is provided, the default type is 'DOMContentLoaded'.
   *
   * @param array $data Associative array of trigger data.
   * @since 1.1.0
   **/
  public function __construct(array $data = [])
  {
    $this->id = (string)($data['id'] ?? 'tri_' . uniqid());
    $this->type = (string)($data['type'] ?? 'DOMContentLoaded');
    $this->setAction((string)($data['action'] ?? ''));
  }

  /**
   * Sets the action of the trigger.
   *
   * If the action is not valid, the action will be set to 'restart'.
   *
   * @param string $action The action to set.
   * @since 1.1.0
   */
  private function setAction(string $action)
  {
    $this->action = TriggerAction::tryFrom($action);
  }

  /**
   * Returns the ID of the trigger.
   *
   * The ID is a unique identifier for the trigger. If no ID was provided when the trigger was created,
   * a unique ID will be generated.
   *
   * @return string The ID of the trigger.
   * @since 1.1.0
   */
  public function getId(): string
  {
    return $this->id;
  }


  /**
   * Returns the type of the trigger.
   *
   * The type is the type of the event that is triggered, such as 'click', 'hover', etc.
   *
   * @return string The type of the trigger.
   * @since 1.1.0
   * */
  public function getType(): string
  {
    return $this->type;
  }


  /**
   * Returns the action of the trigger.
   *
   * The action is the action that is executed when the trigger is triggered.
   *
   * @return string The action of the trigger.
   * @since 1.1.0
   */
  public function getAction(): string
  {
    return $this->action;
  }
}


/**
 * Class representing trigger action types.
 *
 * @package FrontisInteraction.Models\Triggers
 * @since 1.1.0
 */
class TriggerAction
{
  const PLAY = 'play';
  const PAUSE = 'pause';

  /**
   * Try to get a valid action from a string value.
   *
   * @param string $value The value to check.
   * @return string The valid action or PLAY as default.
   */
  public static function tryFrom(string $value): string
  {
    $validActions = [self::PLAY, self::PAUSE];
    return in_array($value, $validActions, true) ? $value : self::PLAY;
  }
}

