<?php

namespace FrontisInteraction\Models;

defined('ABSPATH') || exit;

/**
 * Class representing the trigger point of an interaction.
 *
 * This class handles the user input data for when an interaction should be triggered.
 *
 * @package FrontisInteraction\Models
 * @since 1.1.0
 */
class TriggerPoint
{
   /**
    * Wether the interaction should be triggered on desktop.

    * @var bool
    */
   private bool $desktop;

   /**
    * Wether the interaction should be triggered on tablet.
    * 
    * @var bool
    */
   private bool $tablet;

   /**
    * Wether the interaction should be triggered on mobile landscape.
    * 
    * @var bool
    */
   private bool $mobileLandscape;

   /**
    * Wether the interaction should be triggered on mobile portrait.
    * 
    * @var bool
    */
   private bool $mobilePortrait;


   /**
    * Constructs a new TriggerPoint object.
    *
    * The object is constructed with an associative array of data, where the keys
    * are the names of the screens (desktop, tablet, mobileLandscape, mobilePortrait)
    * and the values are whether the interaction should be triggered on the given screen.
    *
    * If a screen is not given in the data array, the value will be set to true.
    *
    * @param array $data Associative array of screen names and whether the interaction should be triggered on the given screen.
    */
   public function __construct(array $data)
   {
      $this->setScreenBreakpoint('desktop', $data['desktop'] ?? true);
      $this->setScreenBreakpoint('tablet', $data['tablet'] ?? true);
      $this->setScreenBreakpoint('mobileLandscape', $data['mobileLandscape'] ?? true);
      $this->setScreenBreakpoint('mobilePortrait', $data['mobilePortrait'] ?? true);
   }


   /**
    * Sets whether the interaction should be triggered on a given screen.
    *
    * @param string $screen The screen to set (desktop, tablet, mobileLandscape, mobilePortrait).
    * @param bool $enabled Whether the interaction should be triggered on the given screen.
    */
   private function setScreenBreakpoint(string $screen, bool $enabled)
   {
      switch ($screen) {
         case 'desktop':
            $this->desktop = $enabled;
            break;
         case 'tablet':
            $this->tablet = $enabled;
            break;
         case 'mobileLandscape':
            $this->mobileLandscape = $enabled;
            break;
         case 'mobilePortrait':
            $this->mobilePortrait = $enabled;
            break;
      }
   }

   /**
    * Checks if the interaction is enabled on desktop screens.
    *
    * @return bool True if the interaction is enabled on desktop screens, false otherwise.
    */
   public function isEnabledInDesktop()
   {
      return $this->desktop;
   }

   /**
    * Checks if the interaction is enabled on tablet screens.
    *
    * @return bool True if the interaction is enabled on tablet screens, false otherwise.
    */
   public function isEnabledInTablet()
   {
      return $this->tablet;
   }

   /**
    * Checks if the interaction is enabled on mobile landscape screens.
    *
    * @return bool True if the interaction is enabled on mobile landscape screens, false otherwise.
    * */
   public function isEnabledInMobileLandscape()
   {
      return $this->mobileLandscape;
   }

   /**
    * Checks if the interaction is enabled on mobile portrait screens.
    *
    * @return bool True if the interaction is enabled on mobile portrait screens, false otherwise.
    */
   public function isEnabledInMobilePortrait()
   {
      return $this->mobilePortrait;
   }

   /**
    * Checks if all screen types (desktop, tablet, mobileLandscape, mobilePortrait) are enabled.
    *
    * @return bool True if all screen types are enabled, false otherwise.
    */
   public function isEnabledInAllScreen()
   {
      return $this->desktop && $this->tablet && $this->mobileLandscape && $this->mobilePortrait;
   }

   /**
    * Returns a JSON representation of the TriggerPoint object.
    *
    * The JSON representation is an object with four properties:
    * - desktop: a boolean indicating whether the interaction should be triggered on desktop screens.
    * - tablet: a boolean indicating whether the interaction should be triggered on tablet screens.
    * - mobileLandscape: a boolean indicating whether the interaction should be triggered on mobile landscape screens.
    * - mobilePortrait: a boolean indicating whether the interaction should be triggered on mobile portrait screens.
    *
    * @return string A JSON representation of the TriggerPoint object.
    */
   public function toJson()
   {
      return json_encode([
         'desktop' => $this->desktop,
         'tablet' => $this->tablet,
         'mobileLandscape' => $this->mobileLandscape,
         'mobilePortrait' => $this->mobilePortrait,
      ]);
   }
}
