<?php

namespace FrontisInteraction\Models;

defined('ABSPATH') || exit;

/**
 * Class representing a target element in the DOM.
 *
 * This class is responsible for storing and retrieving information about a target element,
 * such as its type (block, element_id, class, etc.) and its value (the specific block name,
 * element ID, class name, etc.).
 *
 * @package FrontisInteraction\Models
 * @since 1.1.0
 */
class Target
{
    /** 
     * Target element selector type
     * 
     * @var string 'block' | 'block_name' | 'element_id' | 'class' | 'css_selector' | 'triggered'
     */
    private string $type;

    /** 
     * Target element selector value
     * 
     * @var string
     */
    private string $value;

    /** 
     *  Selected gutenberg block name
     * 
     * @var string|null
     */
    private ?string $blockName;


    /**
     * Constructor.
     *
     * Creates a new Target object with the given data.
     *
     * @param array $data Associative array of target data.
     * @param string $data['type'] Target element selector type.
     * @param string $data['value'] Target element selector value.
     * @param string|null $data['blockName'] Selected gutenberg block name.
     */
    public function __construct(array $data)
    {

        $this->type = $data['type'] ?? '';
        $this->value = $data['value'] ?? '';
        $this->blockName = $data['blockName'] ?? null;
    }


    /**
     * Retrieves the target element selector type.
     *
     * @return string 'block' | 'block_name' | 'id' | 'class' | 'css_selector' | 'triggered'
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * Retrieves the target element selector value.
     *
     * @return string Target element selector value.
     */
    public function getValue(): string
    {
        return $this->value;
    }

    /**
     * Retrieves the selected Gutenberg block name.
     *
     * If the target was created from a Gutenberg block, this will be the name of that block.
     * If the target was created from a different source, this will be null.
     *
     * @return string|null The selected Gutenberg block name, or null if the target was created from a different source.
     */
    public function getBlockName(): ?string
    {
        return $this->blockName;
    }


    /**
     * Retrieves the target element selector as a string.
     *
     * This function takes into account the target element selector type and value,
     * and returns a string that represents the target element selector.
     *
     * @return string The target element selector as a string.
     */
    public function getSelector(): string
    {
        $type = $this->getType();
        $value = trim($this->getValue());

        switch ($type) {
            case 'block':
                return $this->getBlockSelector($value);
            case 'id':
                return $this->getElementIdSelector($value);
            case 'class':
                return $this->getClassSelector($value);
            case 'css_selector':
                return $this->getCssSelector($value);
            case 'triggered':
                return 'e.currentTarget';
            case 'block_name':
                return '';
            default:
                return "''";
        }
    }


    /**
     * Retrieves a CSS selector string for a Gutenberg block.
     *
     * This function takes a block ID as its parameter and returns a CSS selector string
     * that matches elements with a "data-fbi-id" attribute equal to that block ID.
     *
     * @param string $value The block ID to generate a CSS selector for.
     * @return string A CSS selector string that matches elements with the given block ID.
     */
    private function getBlockSelector(string $value): string
    {
        $sanitized = $this->sanitizeAttr($value);
        return "'[data-fbi-id=\'{$sanitized}\']'";
    }


    /**
     * Retrieves a CSS selector string for an element with a given ID.
     *
     * This function takes an element ID as its parameter and returns a CSS selector string
     * that matches elements with an "id" attribute equal to that element ID.
     *
     * If the element ID is invalid, an empty string is returned.
     *
     * @param string $value The element ID to generate a CSS selector for.
     * @return string A CSS selector string that matches elements with the given element ID.
     */
    private function getElementIdSelector(string $value): string
    {
        $sanitized = $this->sanitizePart($value);
        if ($sanitized === null) {
            return '';
        }
        return "'#{$sanitized}'";
    }



    /**
     * Retrieves a CSS selector string for a given class name.
     *
     * This function takes a class name as its parameter and returns a CSS selector string
     * that matches elements with the given class name.
     *
     * The function splits the class name by spaces and sanitizes each part to ensure that it is
     * a valid CSS selector. It then joins the sanitized parts back together with a dot separator.
     *
     * If the class name is invalid, an empty string is returned.
     *
     * @param string $value The class name to generate a CSS selector for.
     * @return string A CSS selector string that matches elements with the given class name.
     */
    private function getClassSelector(string $value): string
    {
        $parts = preg_split('/\s+/', $value);
        $sanitizedParts = array_filter(array_map([$this, 'sanitizePart'], $parts));
        return !empty($sanitizedParts) ? "'." . implode('.', $sanitizedParts) . "'" : "''";
    }

    /**
     * Improved getCssSelector to handle complex selectors without breaking.
     */
    private function getCssSelector(string $value): string
    {
        $value = trim($value);
        if ($value === '') {
            return "''";
        }

        // Escape single quotes in attribute selectors
        $value = preg_replace_callback('/\[[^\]]+\]/', function ($matches) {
            return str_replace("'", "\\'", $matches[0]);
        }, $value);

        return "'{$value}'";
    }

    /**
     * Sanitizes a single part of a CSS selector.
     *
     * Allows letters, numbers, hyphens, underscores, colons, and escapes.
     * Returns null if empty.
     */
    private function sanitizePart(string $part): ?string
    {
        $part = trim($part);
        if ($part === '') {
            return null;
        }
        // Keep valid characters for IDs/classes, allow escaping special chars
        $part = preg_replace('/[^a-zA-Z0-9\-_]/', '', $part);
        return $part === '' ? null : $part;
    }


    /**
     * Sanitizes an attribute value.
     *
     * Escapes single quotes but leaves other characters intact.
     */
    private function sanitizeAttr(string $value): string
    {
        $value = trim($value);
        $value = str_replace("'", "\\'", $value);
        return $value;
    }
}
