<?php

namespace FrontisInteraction\Models\Sequence;

defined('ABSPATH') || exit;
/**
 * Class for representing the options for a timeline sequence.
 *
 * This class contains the options for a timeline sequence, such as the duration.
 *
 * @package FrontisInteraction\Models\Sequence
 * @since 1.1.0
 */
class SequenceOptions
{
    /**
     * The duration of the sequence
     * 
     * @var float
     */
    private float $duration = 0.25;

    /**
     * The ease of the sequence
     * 
     * @var string
     */
    private string $ease = 'linear';

    /**
     * The timeline position control
     * 
     * @var string|float
     */
    private $at = 0;

    /**
     * The repeat of the sequence
     * 
     * @var int|string
     */
    private $repeat = 0;

    /**
     * The repeat of the sequence
     * 
     * @var string 'loop' | 'reverse'
     */
    private string $repeatType = 'loop';

    /**
     * Initializes the sequence options object with the given data.
     *
     * If a property is not found in the given data, it will be set to its default value.
     *
     * @param array $data Associative array of data.
     */
    public function __construct(array $data = [])
    {
        $this->setDuration($data['duration'] ?? $this->duration);
        $this->setEase($data['ease'] ?? $this->ease);
        $this->setAt($data['at'] ?? $this->at);
        $this->setRepeat($data['repeat'] ?? $this->repeat);
        $this->setRepeatType($data['repeatType'] ?? $this->repeatType);
    }

    /**
     * Sanitizes an integer value.
     * 
     * If the value is not a valid integer, it will be set to 0.
     * If the value is less than the given minimum, it will be set to the minimum.
     * 
     * @param mixed $val The value to sanitize.
     * @param int $min The minimum value to return if the sanitized value is less than the minimum.
     * @return int The sanitized value.
     */
    private function sanitizeInt($val, $min = 0)
    {
        $val = filter_var($val, FILTER_VALIDATE_INT, ["options" => ["default" => 0]]);
        return max($min, $val);
    }

    /**
     * Sanitizes a float value.
     *
     * If the value is not a valid float, it will be set to 0.0.
     * If the value is less than the given minimum, it will be set to the minimum.
     *
     * @param mixed $val The value to sanitize.
     * @param float $min The minimum value to return if the sanitized value is less than the minimum.
     * @return float The sanitized value.
     */
    private function sanitizeFloat($val, $min = 0.0)
    {
        $val = is_numeric($val) ? (float)$val : 0.0;
        return max($min, $val);
    }

    /**
     * Sanitizes a boolean value.
     * 
     * If the value is not a valid boolean, it will be set to false.
     * 
     * @param mixed $val The value to sanitize.
     * @return bool The sanitized value.
     */
    private function sanitizeBool($val)
    {
        return filter_var($val, FILTER_VALIDATE_BOOL, FILTER_NULL_ON_FAILURE) ?? false;
    }

    /**
     * Sanitizes an ease value by removing any characters that are not alphanumeric, dot, hyphen, parentheses, or comma.
     * If the sanitized value is empty, it will be set to 'linear'.
     * 
     * @param string $ease The ease value to sanitize.
     * @return string The sanitized ease value.
     */
    private function sanitizeEase(string $ease): string
    {
        return preg_replace('/[^a-zA-Z0-9\.\-\(\),]/', '', $ease) ?: 'linear';
    }

    /**
     * Sanitizes an 'at' value, which can be a position label or a number.
     * 
     * If the value is a number, it will be returned as an integer.
     * If the value is a string, any characters that are not alphanumeric, dot, hyphen, parentheses, or comma will be removed.
     * 
     * @param mixed $val The value to sanitize.
     * @return string|float The sanitized value.
     */
    private function sanitizeAt($val)
    {
        // can be position label or number `+=0.5`
        if (is_numeric($val)) return (float)$val;

        // sanitize but allow "+=, -=, <+= etc"
        return preg_replace('/[^a-zA-Z0-9\+\-\.\<\>]/', '', (string)$val);
    }


    /**
     * Sets the duration of the sequence.
     *
     * The duration is the length of the sequence in seconds.
     * A value of 0 will result in an instant sequence.
     *
     * @param mixed $val The duration value to set.
     * @return self The instance of this class.
     */
    private function setDuration($val)
    {
        $this->duration = $this->sanitizeFloat($val, 0);
        return $this;
    }

    /**
     * Sets the easing of the sequence.
     *
     * The easing is the type of easing used by the sequence.
     * Examples of easing types include 'easeInQuad', 'easeOutQuad', 'easeInOutQuad', and 'none'.
     *
     * @param string $val The easing type to set.
     * @return self The instance of this class.
     */
    private function setEase(string $val): self
    {
        $this->ease = $this->sanitizeEase($val);
        return $this;
    }

    /**
     * Sets the 'at' value of the sequence.
     * 
     * The 'at' value determines when the sequence will start.
     * It can be a position label (e.g., "start", "end", "50%") or a number (e.g., 0, 1, 2).
     * If the value is a number, it represents the position of the sequence in seconds.
     * If the value is a string, it will be parsed as a position label.
     * 
     * @param mixed $val The 'at' value to set.
     * @return self The instance of this class.
     */
    private function setAt($val)
    {
        $this->at = $this->sanitizeAt($val);
        return $this;
    }

    /**
     * Sets the number of times the sequence should repeat.
     * 
     * A value of 0 will result in no repeat.
     * A value of 1 or greater will result in the sequence repeating the given number of times.
     * 
     * @param mixed $val The number of times to repeat the sequence.
     * @return self The instance of this class.
     **/
    private function setRepeat($val)
    {
        $this->repeat = $this->sanitizeInt($val, -1);
        return $this;
    }


    /**
     * Sets the repeat type of the sequence.
     * 
     * The repeat type determines how the sequence will repeat.
     * If the value is 'reverse', the sequence will repeat in reverse order.
     * If the value is not 'reverse', the sequence will repeat in loop order.
     * 
     * @param mixed $val The repeat type to set. Must be either 'reverse' or not 'reverse'.
     * @return self The instance of this class.
     */
    private function setRepeatType($val)
    {
        if ($val == 'reverse') {
            $this->repeatType = 'reverse';
        } else {
            $this->repeatType = 'loop';
        }
        return $this;
    }

    /**
     * Returns the duration of the sequence in seconds.
     *
     * @return float The duration of the sequence in seconds.
     */
    public function getDuration(): float
    {
        return $this->duration;
    }

    /**
     * Returns the ease of the sequence.
     *
     * The ease is a string that represents the ease of the sequence.
     * It can be one of the following: "power1.in", "power1.out", "power1.inOut", "power2.in", "power2.out", "power2.inOut", "power3.in", "power3.out", "power3.inOut", "power4.in", "power4.out", "power4.inOut", "linear", "easeInQuad", "easeOutQuad", "easeInOutQuad", "easeInCubic", "easeOutCubic", "easeInOutCubic", "easeInQuart", "easeOutQuart", "easeInOutQuart", "easeInQuint", "easeOutQuint", "easeInOutQuint", "easeInSine", "easeOutSine", "easeInOutSine", "easeInExpo", "easeOutExpo", "easeInOutExpo", "easeInCirc", "easeOutCirc", "easeInOutCirc", "easeInElastic", "easeOutElastic", "easeInOutElastic", "easeInBack", "easeOutBack", "easeInOutBack", "easeInBounce", "easeOutBounce", "easeInOutBounce".
     *
     * @return string The ease of the sequence.
     * */
    public function getEase(): string
    {
        return $this->ease;
    }

    /**
     * Returns the position of the sequence in the timeline.
     *
     * If the sequence is set to start at a specific position in the timeline,
     * this function will return that position as a string or an integer.
     * If the sequence is set to start at the beginning or end of the timeline,
     * this function will return the corresponding string.
     *
     * @return string|float The position of the sequence in the timeline.
     */
    public function getStartAt()
    {
        return $this->at;
    }

    /**
     * Returns the number of times the sequence should repeat.
     *
     * The repeat value can be any non-negative integer. If the repeat value is 0,
     * the sequence will not repeat at all. If the repeat value is greater than 0,
     * the sequence will repeat the specified number of times.
     *
     * @return int The number of times the sequence should repeat.
     * 
     **/
    public function getRepeat(): int
    {
        return $this->repeat;
    }

    /**
     * Returns whether the sequence should reverse itself on repeat.
     *
     * If the "repeat" property is set to a non-zero value, the "yoyo" property will be set to the value of the $yoyo parameter, cast to a boolean.
     * If the "repeat" property is set to zero, the "yoyo" property will be set to null.
     *
     * @return bool Whether the sequence should reverse itself on repeat.
     */
    public function getRepeatType(): string
    {
        return $this->repeatType;
    }

    /**
     * Returns an array representation of the sequence's properties.
     * 
     * The returned array will contain the following properties:
     * - duration: the duration of the sequence in seconds
     * - ease: the ease of the sequence
     * - repeat: the number of times the sequence should repeat
     * - repeatType: whether the sequence should reverse itself on repeat
     * 
     * @return array An array representation of the sequence's properties.
     */
    public function toArray(): array
    {
        return [
            'duration' => $this->duration,
            'ease'     => $this->ease,
            'repeat'   => $this->repeat,
            'repeatType' => $this->repeatType,
        ];
    }
}
