<?php

namespace FrontisInteraction\Models\Sequence;

use FrontisInteraction\Models\Factories\SequencePropertyFactory;
use FrontisInteraction\Models\Sequence\SequenceOptions;
use FrontisInteraction\Models\Target;

defined('ABSPATH') || exit;

/**
 * Class representing a timeline Sequence.
 *
 * @package FrontisInteraction\Models\Sequence
 * @since 1.1.0
 **/
class Sequence
{
    /**
     * The id of the sequence
     * 
     * @var string
     */
    private string $id;

    /**
     * The active state of the sequence
     * 
     * @var bool
     */
    private bool $active;

    /** 
     * The target of the sequence
     * 
     * @var Target
     * */
    private Target $target;

    /**
     * The type of the sequence
     * 
     * @var array 'from' | 'to'
     */
    private array $type;

    /**
     * The from sequence properties
     * 
     * @var array
     */
    private array $from;

    /**
     * The to sequence properties
     * 
     * @var array
     */
    private array $to;


    /**
     * The sequence configuration options and properties
     * 
     * @var SequenceOptions
     */
    private SequenceOptions $options;


    /**
     * Constructor.
     *
     * Initializes the sequence object with the given data.
     *
     * @param array $data Associative array of data.
     *
     * @since 1.1.0
     */
    public function __construct(array $data)
    {
        $this->id = (string) ($data['id'] ?? 'ia_' . uniqid());

        $this->target = new Target($data['target'] ?? []);

        $this->setSequenceType($data['type'] ?? []);
        $this->setSequence('from', $data['from'] ?? []);
        $this->setSequence('to', $data['to'] ?? []);
        $this->setSequenceOptions($data['options'] ?? []);
    }




    public function setSequenceType(array $type)
    {
        if (array_search('from', $type) !== false && array_search('to', $type) !== false) {
            $this->type = ['from', 'to'];
        } else {
            $this->type = ['to'];
        }
    }

    /**
     * Set a sequence property by its key.
     * 
     * The function will loop through the given array and sanitize each value using the SequencePropertyFactory.
     * 
     * @param string $sequenceKey The key for the sequence property (e.g., "from", "to").
     * @param array $sequence The sequence property to set.
     * @since 1.1.0
     */
    private function setSequence(string $sequenceKey, array $sequence)
    {
        $sanitizedSequence = [];
        foreach ($sequence as $key => $value)
            $sanitizedSequence[$key] = SequencePropertyFactory::create($key, $value);

        $this->{$sequenceKey} = $sanitizedSequence;
    }


    private function setSequenceOptions(array $options)
    {
        $this->options = new SequenceOptions($options);
    }

    /**
     * Returns the ID of the timeline sequence.
     *
     * The ID is a unique identifier for the timeline sequence and is used to identify the timeline sequence in the interaction editor.
     *
     * @return string The ID of the timeline sequence.
     * @since 1.1.0
     */
    public function getId(): string
    {
        return $this->id;
    }


    /**
     * Returns the types of the sequence.
     *
     * The types are an array of strings containing the type of the sequence (e.g., "from", "to").
     *
     * @return array The types of the sequence.
     * @since 1.1.0
     */
    public function getTypes(): array
    {
        return $this->type;
    }

    /**
     * Returns the target element of the sequence.
     *
     * The target element is the element that will be affected by the sequence.
     *
     * @return Target The target element of the sequence.
     * @since 1.1.0
     */
    public function getTarget(): Target
    {
        return $this->target;
    }

    public function getProperties(): array
    {
        $properties = [];
        if (count($this->getTypes()) === 2) {
            foreach ($this->from as $key => $value) {
                $properties[$key] = [$value, $this->to[$key]];
            }
        } else {
            $properties = $this->to;
        }
        return $properties;
    }


    /**
     * Returns the sequence options of the timeline sequence.
     *
     * The sequence options are an object containing the configuration options for the timeline sequence,
     * such as the duration, easing, and delay.
     *
     * @return SequenceOptions The sequence options of the timeline sequence.
     * @since 1.1.0
     */
    public function getOptions(): SequenceOptions
    {
        return $this->options;
    }
}
