<?php

namespace FrontisInteraction\Models\Sequence\Property;

defined('ABSPATH') || exit;

/**
 * Class representing a unit property.
 *
 * A unit property is a property that has a value that can be measured in different units.
 * For example, a length property can have a value of 100px, 100cm, or 3.9in.
 * The unit property will store the value with the unit, and provide a method to get the value with a different unit.
 *
 * @package FrontisInteraction\Models\Sequence\property
 * @since 1.1.0
 **/
class UnitProperty implements BaseProperty
{
    /**
     * @var mixed
     */
    private $value;

    /**
     * @var array
     */
    private $units;

    /**
     * @var float|int
     */
    private $default;

    /**
     * Construct a new UnitProperty instance.
     *
     * @param mixed $value the value of the property. Can be a string, float, int, or null.
     * @param array $units the possible units of the property. Can be empty if no units apply.
     * @param float|int $default the default value if no unit is applicable. Defaults to 0.
     */
    public function __construct($value, $units, $default = 0)
    {
        $this->value = $value;
        $this->units = $units;
        $this->default = $default;
    }

    /**
     * Get the value of the property, with unit if applicable.
     *
     * If the value is numeric, it will be returned as a float/int.
     * If the value is a string with a number followed by a unit, it will be returned as a string with the unit.
     * If the value is a string with just a number, it will be returned as a float/int with the default unit if applicable.
     * If the value is a string with no number, it will be returned as the default value with the default unit if applicable.
     *
     * @return string|float|int
     */
    public function getValue()
    {
        $units      = $this->units;
        $hasUnits   = !empty($units);
        $defaultUnit = $hasUnits ? $units[0] : null;

        if (is_numeric($this->value)) {
            return $hasUnits
                ? $this->value . $defaultUnit
                : (float) $this->value;
        }

        $valueStr = trim((string) $this->value);

        if (preg_match('/^(-?\d+(?:\.\d+)?)\s*([a-z%]*)$/i', $valueStr, $matches)) {
            $number = (float) $matches[1];
            $unit   = strtolower($matches[2]);

            if ($unit === '') {
                return $defaultUnit !== null
                    ? $number . $defaultUnit
                    : $number;
            }

            if (in_array($unit, $units, true)) {
                return $number . $unit;
            }

            return $defaultUnit !== null
                ? $number . $defaultUnit
                : $number;
        }

        return $defaultUnit !== null
            ? $this->default . $defaultUnit
            : $this->default;
    }
}
