<?php

namespace FrontisInteraction\Models\Sequence\Property;

defined('ABSPATH') || exit;

/**
 * Class representing a text property.
 *
 * A text property is a property that has a string value.
 * The property will store the value, and provide a method to get the value.
 *
 * @package FrontisInteraction\Models\Sequence\Property
 * @since 1.1.0
 */
class TextProperty implements BaseProperty
{
    /**
     * @var mixed
     */
    private $value;

    /**
     * @var string
     */
    private $default;

    /**
     * Construct a new TextProperty instance.
     *
     * @param mixed $value the value of the property. Can be a string, float, int, or null.
     * @param string $default the default value if no unit is applicable. Defaults to an empty string.
     */
    public function __construct($value, $default = '')
    {
        $this->value = $value;
        $this->default = $default;
    }

    /**
     * Gets the value of the property.
     *
     * If the value is null or empty, returns the default value.
     * If the value is a scalar or an object with a __toString method, casts it to a string.
     * If the value is anything else, returns the default value.
     *
     * @return string The value of the property, or the default value if the value is not valid.
     */
    public function getValue(): string
    {
        if ($this->value === null || $this->value === '') {
            return $this->default;
        }

        if (is_scalar($this->value) || (is_object($this->value) && method_exists($this->value, '__toString'))) {
            return (string) $this->value;
        }

        return $this->default;
    }
}
