<?php

namespace FrontisInteraction\Models\Sequence\Property;

defined('ABSPATH') || exit;

/**
 * Class representing an opacity property.
 *
 * An opacity property is a property that has a float value between 0 and 1, representing the opacity of the element.
 *
 * @package FrontisInteraction\Models\Sequence\Property
 * @since 1.1.0
 */
class OpacityProperty implements BaseProperty
{
    /**
     * @var mixed
     */
    private $value;

    /**
     * @var int
     */
    private $default;

    /**
     * Construct a new OpacityProperty instance.
     *
     * @param mixed $value the value of the property. Can be a string, float, int, or null.
     * @param int $default the default value if no unit is applicable. Defaults to 100.
     */
    public function __construct($value, $default = 100)
    {
        $this->value = $value;
        $this->default = $default;
    }

    /**
     * Get the value of the property.
     *
     * This method will return a float value between 0 and 1, representing the opacity of the element.
     * If the value is null, empty, or not a number, the default value will be returned.
     * If the value is a number, it will be cast to an int and then normalized to a value between 0 and 100.
     * The normalized value will then be divided by 100 to get a float value between 0 and 1.
     *
     * @return int|float
     */
    public function getValue()
    {
        $raw = $this->normalizeToInt($this->value, $this->default);

        $raw = max(0, min(100, $raw));
        $fraction = $raw / 100;

        return round($fraction, 2);
    }

    /**
     * Normalize a value to an integer.
     *
     * If the value is null, empty, or not a number, return the default value.
     * Otherwise, return the value cast to an int.
     *
     * @param mixed $value the value to normalize.
     * @param int $default the default value to return if the value is invalid.
     * @return int
     */
    private function normalizeToInt($value, $default)
    {
        if ($value === null || $value === '' || !is_numeric($value)) {
            return $default;
        }

        return (int) $value;
    }
}
