<?php

namespace FrontisInteraction\Models\Sequence\Property;

defined('ABSPATH') || exit;

/**
 * Class representing a numeric property.
 *
 * A numeric property is a property that can have a decimal value.
 * The value will be rounded to a maximum of 2 decimal places.
 *
 * @package FrontisInteraction\Models\Sequence\Property
 * @since 1.1.0
 */
class NumericProperty implements BaseProperty
{
    /**
     * @var mixed
     */
    private $value;

    /**
     * @var float
     */
    private $default;

    /**
     * Construct a new NumericProperty instance.
     *
     * @param mixed $value The value of the property. Can be a string, float, int, or null.
     * @param float $default The default value if no valid numeric value is provided. Defaults to 0.0.
     */
    public function __construct($value, $default = 0.0)
    {
        $this->value = $value;
        $this->default = $default;
    }

    /**
     * Get the value of the property.
     *
     * If the value is null or empty, return the default value.
     * If the value is numeric, cast it to float and round to 2 decimal places.
     * Otherwise, return the default value.
     *
     * @return float
     */
    public function getValue(): float
    {
        if ($this->value === null || $this->value === '') {
            return $this->default;
        }

        if (is_numeric($this->value)) {
            return round((float) $this->value, 2);
        }

        return $this->default;
    }
}
