<?php

namespace FrontisInteraction\Models\Sequence\Property;

defined('ABSPATH') || exit;

/**
 * Class representing a box shadow property.
 *
 * A box shadow property is a property that has a value with six elements, representing the horizontal shadow, vertical shadow, shadow blur, shadow spread, shadow color and the position.
 *
 * The property will store the value, and provide a method to get the value.
 *
 * @package FrontisInteraction\Models\Sequence\Property
 * @since 1.1.0
 */
class BoxShadowProperty implements BaseProperty
{
    /**
     * @var mixed
     */
    private $value;

    /**
     * @var array
     */
    private $default;

    public function __construct(
        $value,
        $default = [
            'x' => '0px',
            'y' => '0px',
            'blur' => '0px',
            'spread' => '0px',
            'color' => '#000000',
            'inset' => false,
        ]
    ) {
        $this->value = $value;
        $this->default = $default;
    }

    /**
     * Get the value of the property as a string.
     *
     * The method normalizes the property value and returns it as a string.
     * The string is in the format of: 'inset x y blur spread color'.
     * If the property value is not an array, an empty array is used.
     * If the property value does not contain a certain key, the default value for that key is used.
     *
     * @return string The value of the property as a string.
     */
    public function getValue(): string
    {
        $val = $this->normalize($this->value);

        $inset = !empty($val['inset']) ? 'inset ' : '';

        return "{$inset}{$val['x']} {$val['y']} {$val['blur']} {$val['spread']} {$val['color']}";
    }

    /**
     * Normalize the property value.
     *
     * The method takes a mixed value as input. If the value is not an array, an empty array is used.
     * The method then normalizes the property value by using the default values for the missing keys.
     *
     * The normalized property value is returned as an array with the keys 'x', 'y', 'blur', 'spread', 'color', and 'inset'.
     *
     * @param mixed $value The value to normalize.
     * @return array The normalized property value.
     */
    private function normalize($value)
    {
        if (!is_array($value)) {
            $value = [];
        }

        return [
            'x' => $this->toString($value['x'] ?? $this->default['x']),
            'y' => $this->toString($value['y'] ?? $this->default['y']),
            'blur' => $this->toString($value['blur'] ?? $this->default['blur']),
            'spread' => $this->toString($value['spread'] ?? $this->default['spread']),
            'color' => $this->toString($value['color'] ?? $this->default['color']),
            'inset' => !empty($value['inset']) ? true : false,
        ];
    }

    /**
     * Converts a mixed value to a string.
     *
     * If the value is scalar (int, float, string, bool), it is cast to a string.
     * If the value is anything else, an empty string is returned.
     *
     * @param mixed $value The value to convert.
     * @return string The converted value.
     */
    private function toString($value)
    {
        return is_scalar($value) ? (string) $value : '';
    }
}
