<?php

namespace FrontisInteraction\Models\Sequence\Property;

defined('ABSPATH') || exit;

/**
 * BorderProperty class.
 *
 * Class representing a border property.
 * A border property is a property that has a value with three elements, representing the width, style, and color of the border.
 * The property will store the value, and provide a method to get the value as a string.
 *
 * @package FrontisInteraction\Models\Sequence\Property
 * @since 1.1.0
 */
class BorderProperty implements BaseProperty
{
    /**
     * @var mixed
     */
    private $value;

    /**
     * @var array
     */
    private $default;

    /**
     * Constructs a new BorderProperty instance.
     *
     * The constructor takes a mixed value, which can be a string, float, int, or null.
     * The value is the value of the border property, and it can be in the format of 'width style color'.
     * If the value is not an array, an empty array is used.
     * The constructor also takes an optional array of default values.
     * The default values are used if the value does not contain a certain key.
     * The default values are: 'width' => '0px', 'style' => 'solid', 'color' => '#000000'.
     *
     * @param mixed $value The value of the border property.
     * @param array $default The default values of the border property. Defaults to [
     *     'width' => '0px',
     *     'style' => 'solid',
     *     'color' => '#000000',
     * ].
     */
    public function __construct(
        $value,
        $default = [
            'width' => '0px',
            'style' => 'solid',
            'color' => '#000000',
        ]
    ) {
        $this->value = $value;
        $this->default = $default;
    }

    /**
     * Returns the value of the property as a string.
     * 
     * The method normalizes the property value and returns it as a string.
     * The string is in the format of: 'width style color'.
     * If the property value is not an array, an empty array is used.
     * If the property value does not contain a certain key, the default value for that key is used.
     * 
     * @return string The value of the property as a string.
     */
    public function getValue(): string
    {
        $val = $this->normalize($this->value);

        return "{$val['width']} {$val['style']} {$val['color']}";
    }

    /**
     * Normalize the given value for border property.
     * 
     * The function takes a mixed value, checks if it's an array, and if not, sets it as an empty array.
     * It then takes the 'width', 'style', and 'color' keys from the array and sets them to the default value
     * if not set.
     * Finally, it returns an array with the normalized 'width', 'style', and 'color' values.
     * 
     * @param mixed $value The value to normalize.
     * @return array The normalized value.
     */
    private function normalize($value)
    {
        if (!is_array($value)) {
            $value = [];
        }

        return [
            'width' => $this->toString($value['width'] ?? $this->default['width']),
            'style' => $this->toString($value['style'] ?? $this->default['style']),
            'color' => $this->toString($value['color'] ?? $this->default['color']),
        ];
    }

    /**
     * Converts a mixed value to a string.
     *
     * If the value is scalar (int, float, string, bool), it is cast to a string.
     * If the value is anything else, an empty string is returned.
     *
     * @param mixed $value The value to convert.
     * @return string The converted value.
     */
    private function toString($value)
    {
        return is_scalar($value) ? (string) $value : '';
    }
}
