<?php

namespace FrontisInteraction\Models\Options;

defined('ABSPATH') || exit;

/**
 * Class representing the options for the load trigger event.
 *
 * This class contains the options for the load trigger event, such as the delay.
 *
 * @package FrontisInteraction\Models\Options
 * @since 1.1.0
 */
class LoadOptions extends TriggerOptions
{
    /**
     * Delay in seconds for the load trigger event.
     * 
     * The delay is the amount of time in seconds to wait before triggering the timeline action after the page load event.
     * 
     * @var float $delay
     * @since 1.1.0
     */
    private float $delay = 0.0;


    /**
     * Constructor.
     *
     * Creates a new LoadOptions object with the given options.
     *
     * @param array $options Associative array of options.
     *
     * The associative array should contain the following keys:
     *   - delay: The delay in seconds for the load trigger event. If not specified, defaults to 0.
     *
     * @since 1.1.0
     */
    public function __construct(array $options)
    {
        parent::__construct($options);
        $this->setDelay($options['delay'] ?? 0);
    }


    /**
     * Sets the delay for the load trigger event.
     *
     * @param float $delay The delay in seconds for the load trigger event.
     *
     * If the given delay is not a number, it will default to 0.
     *
     * The delay will be rounded to 2 decimal places.
     *
     * @since 1.1.0
     */
    private function setDelay($delay): void
    {
        if (!is_numeric($delay)) {
            $this->delay = 0.0;
            return;
        }

        $delay = max(0.0, (float) $delay);
        $this->delay = round($delay, 2);
        $this->delay = round((float) $delay, 2);
    }


    /**
     * Checks if the load trigger event has a delay.
     *
     * If the load trigger event has a delay, this method will return true.
     * Otherwise, it will return false.
     *
     * @return bool True if the load trigger event has a delay, false otherwise.
     *
     * @since 1.1.0
     */
    public function isDelayed(): bool
    {
        return $this->delay > 0;
    }

    /**
     * Gets the delay for the load trigger event in seconds.
     *
     * The delay is the amount of time in seconds to wait before triggering the timeline action after the page load event.
     *
     * @return float The delay for the load trigger event in seconds.
     *
     * @since 1.1.0
     */
    public function getDelayInSeconds(): float
    {
        return $this->delay;
    }

    /**
     * Gets the delay for the load trigger event in milliseconds.
     *
     * @return int The delay for the load trigger event in milliseconds.
     *
     * @since 1.1.0
     */
    public function getDelayInMilliseconds(): int
    {
        return (int) ($this->delay * 1000);
    }
}
