<?php

namespace FrontisInteraction\Models\Options;

defined('ABSPATH') || exit;

/**
 * Class representing the options for a hover trigger event.
 *
 * This class handles the user hover event options data.
 *
 * @package FrontisInteraction\Models\Options
 * @since 1.1.0
 */
class HoverOptions extends TriggerOptions
{
    /**
     * The type of hover event.
     *
     * @var string
     * @since 1.1.0
     */
    private string $type;

    /**
     * Whether to stop the hover event propagation.
     *
     * If true, the hover event will not propagate to parent elements.
     *
     * @var bool
     * @since 1.1.0
     */
    private bool $stopPropagation = false;


    /**
     * Constructor.
     *
     * Creates a new HoverOptions object with the given options.
     *
     * The associative array should contain the following keys:
     *   - stopPropagation: Whether to stop the hover event propagation.
     *   - type: The type of hover event.
     *
     * @param array $options Associative array of options.
     * @since 1.1.0
     */
    public function __construct(array $options)
    {
        parent::__construct($options);

        $this->stopPropagation = $options['stopPropagation'] ?? false;
        $this->setHoverType($options['type'] ?? HoverType::MOUSE_ENTER);
    }


    /**
     * Sets the type of hover event.
     *
     * If the type is invalid, it will default to HoverType::MOUSE_ENTER.
     *
     * @param string $type The type of hover event.
     * @since 1.1.0
     */
    private function setHoverType(string $type)
    {
        $type = HoverType::tryFrom($type ?? '');

        $this->type = $type;
    }

    /**
     * Gets the type of hover event.
     *
     * @return string The type of hover event.
     * @since 1.1.0
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * Whether to stop the hover event propagation.
     *
     * If true, the hover event will not propagate to parent elements.
     *
     * @return bool True if the hover event should be stopped, false otherwise.
     * @since 1.1.0
     */
    public function isStopPropagation(): bool
    {
        return $this->stopPropagation;
    }
}


/**
 * Class representing the type of hover event.
 *
 * HoverType::MOUSE_ENTER - Triggered when the mouse enters the element area.
 * HoverType::MOUSE_LEAVE - Triggered when the mouse leaves the element area.
 * 
 * @package FrontisInteraction\Models\Options
 * @since 1.1.0
 */
class HoverType
{
    const MOUSE_ENTER = 'mouseenter';
    const MOUSE_LEAVE = 'mouseleave';

    /**
     * Try to get a valid hover type from a string value.
     *
     * @param string $value The value to check.
     * @return string The valid hover type or MOUSE_ENTER as default.
     */
    public static function tryFrom(string $value): string
    {
        $validTypes = [self::MOUSE_ENTER, self::MOUSE_LEAVE];
        return in_array($value, $validTypes, true) ? $value : self::MOUSE_ENTER;
    }
}

