<?php

namespace FrontisInteraction\Models\Options;

defined('ABSPATH') || exit;

/**
 * Class representing the options for a click trigger event.
 *
 * This class handles the user click event options data.
 *
 * @package FrontisInteraction\Models\Options
 * @since 1.1.0
 */
class ClickOptions extends TriggerOptions
{
    /**
     * Whether to prevent the default click action.
     * 
     * @var bool
     * @since 1.1.0
     */
    private bool $preventDefault = false;

    /**
     * Whether to stop the click event propagation.
     * 
     * @var bool
     * @since 1.1.0
     */
    private bool $stopPropagation = false;


    /**
     * Constructor.
     *
     * Creates a new ClickOptions object with the given options.
     *
     * Supported options:
     *  * bool $preventDefault = false: Whether to prevent the default click action.
     *  * bool $stopPropagation = false: Whether to stop the click event propagation.
     *
     * @param array $options Associative array of options.
     * @since 1.1.0
     **/

    public function __construct(array $options)
    {
        parent::__construct($options);
        $this->preventDefault = $options['preventDefault'] ?? false;
        $this->stopPropagation = $options['stopPropagation'] ?? false;
    }

    /**
     * Whether to prevent the default click action.
     *
     * @return bool True if the default click action should be prevented, false otherwise.
     * @since 1.1.0
     */
    public function isPreventDefault(): bool
    {
        return $this->preventDefault;
    }

    /**
     * Whether to stop the click event propagation.
     *
     * @return bool True if the click event propagation should be stopped, false otherwise.
     * @since 1.1.0
     */
    public function isStopPropagation(): bool
    {
        return $this->stopPropagation;
    }
}
