<?php

namespace FrontisInteraction\Models;

use FrontisInteraction\Models\Factories\TriggerFactory;
use FrontisInteraction\Models\Sequence\Sequence;
use FrontisInteraction\Models\Triggers\FallbackTrigger;
use FrontisInteraction\Models\Triggers\TriggerInterface;

defined('ABSPATH') || exit;

class Interaction
{
    /**
     * The id of the interaction.
     * 
     * @var string
     */
    private string $id;

    /**
     * Whether the interaction is active or not.
     * 
     * @var bool
     */
    private bool $active;

    /**
     * The triggers of the interaction.
     * 
     * @var TriggerInterface[]
     */
    private array $triggers;

    /**
     * The Sequences of the interaction.
     * 
     * @var Sequences[]
     */
    private array $sequences;

    /**
     * The trigger point of the interaction for viewport breakpoints.
     * 
     * @var TriggerPoint
     */
    private TriggerPoint $triggerPoint;


    public function __construct(array $data)
    {
        $this->id = (string) ($data['id'] ?? 'ia_' . uniqid());
        $this->active = (bool) ($data['active'] ?? false);

        $this->setTriggers($data['triggers'] ?? []);

        $this->sequences = array_map(fn($sequence) => new Sequence($sequence), $data['sequences']);
        $this->triggerPoint = new TriggerPoint($data['triggerPoint'] ?? []);
    }


    /**
     * Sets the triggers of the interaction.
     *
     * @param array $triggers An array of trigger objects.
     */
    private function setTriggers(array $triggers)
    {
        if (empty($triggers)) {
            $this->triggers[] = new FallbackTrigger(['action' => 'play']);
            return;
        }

        foreach ($triggers as $trigger) {
            $this->triggers[] = TriggerFactory::create($trigger);
        }
    }

    /**
     * Gets the id of the interaction.
     *
     * @return string The id of the interaction.
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Checks if the interaction is active.
     *
     * @return bool
     */
    public function isActive()
    {
        return $this->active;
    }


    /**
     * Returns the triggers of the interaction.
     *
     * The triggers are an array of TriggerInterface objects.
     *
     * @return TriggerInterface[] The triggers of the interaction.
     */
    public function getTriggers(): array
    {
        return $this->triggers;
    }

    /**
     * Gets the timeline of the interaction.
     *
     * The timeline is an array of Tween objects.
     *
     * @return Sequence[] The timeline of the interaction.
     */
    public function getSequences()
    {
        return $this->sequences;
    }

    /**
     * Gets the trigger point of the interaction.
     *
     * The trigger point is an object that defines when the interaction should be triggered.
     *
     * @return TriggerPoint The trigger point of the interaction.
     */
    public function getTriggerPoint()
    {
        return $this->triggerPoint;
    }
}
