<?php

namespace FrontisInteraction\Models\Factories;

use FrontisInteraction\Models\Sequence\Property\{
    BorderProperty,
    BoxShadowProperty,
    UnitProperty,
    NumericProperty,
    OpacityProperty,
    TextProperty,
    TextShadowProperty
};

defined('ABSPATH') || exit;

/**
 * Class for generating tween properties.
 *
 * @package FrontisInteraction\Models\Factories
 * @since 1.1.0
 */
class SequencePropertyFactory
{
    /**
     * Generate a tween property based on the given key and value.
     * 
     * The factory will try to match the given key to a specific unit property
     * and return the sanitized value. If no match is found, an empty string
     * will be returned.
     * 
     * @param string $key the key to match against the available unit properties.
     * @param mixed $value the value to sanitize.
     * @return mixed the sanitized value.
     */
    public static function create(string $key, $value)
    {
        switch ($key) {
            case 'x':
            case 'y':
            case 'z':
                return (new UnitProperty($value, ['%', 'px', 'em', 'ch', 'rem', 'vh', 'vw', 'svh', 'svw'], 0))->getValue();
            
            case 'scale':
            case 'scaleX':
            case 'scaleY':
                return (new NumericProperty($value, 1))->getValue();
            
            case 'skewX':
            case 'skewY':
            case 'rotate':
            case 'rotateX':
            case 'rotateY':
            case 'rotateZ':
                return (new NumericProperty($value, 1))->getValue();
            
            case 'opacity':
                return (new OpacityProperty($value, 100))->getValue();
            
            case 'visibility':
            case 'backfaceVisibility':
            case 'color':
            case 'backgroundColor':
                return (new TextProperty($value, 'inherit'))->getValue();
            
            case 'border':
                return (new BorderProperty($value))->getValue();
            
            case 'borderRadius':
                return (new UnitProperty($value, ['%', 'px', 'em', 'ch', 'rem', 'vh', 'vw', 'svh', 'svw'], 0))->getValue();
            
            case 'boxShadow':
                return (new BoxShadowProperty($value))->getValue();
            
            case 'textShadow':
                return (new TextShadowProperty($value))->getValue();
            
            case 'width':
            case 'height':
                return (new UnitProperty($value, ['%', 'px', 'em', 'ch', 'rem', 'vh', 'vw', 'svh', 'svw'], 0))->getValue();
            
            case 'zIndex':
                return (new NumericProperty($value, 0))->getValue();
            
            case 'clipPath':
                return (new TextProperty($value, ''))->getValue();
            
            default:
                return '';
        }
    }
}
