<?php

namespace FrontisInteraction\Generator\Triggers;

use FrontisInteraction\Models\Triggers\LoadTrigger;

defined('ABSPATH') || exit;

/**
 * Class to generate JavaScript code for a Motion One timeline trigger from a given LoadTrigger object.
 *
 * This class handles the user load event options data.
 *
 * @package FrontisInteraction\Generator\Triggers
 */
class LoadEventGenerator implements TriggerEventGenerator
{
    /**
     * The LoadTrigger object to generate code for.
     *
     * @var LoadTrigger
     */
    private LoadTrigger $trigger;

    /**
     * The Animation timeline variable to use in the generated code.
     *
     * @var string
     */
    private string $animationVar;

    /**
     * A callback function for generating triggered element animation code.
     * 
     * @var \Closure
     */
    private \Closure $innerAnimationCB;

    /**
     * Constructor for the LoadEventGenerator class.
     *
     * Initializes the class with the given LoadTrigger object, animation timeline variable, and a callback for generating triggered animation code.
     *
     * @param LoadTrigger $trigger The LoadTrigger object to generate code for.
     * @param string $animationVar The Animation timeline variable to use in the generated code.
     * @param \Closure $innerAnimationCB A callback function for generating triggered animation code.
     * @since 1.1.0
     */
    public function __construct(LoadTrigger $trigger, string $animationVar, \Closure $innerAnimationCB)
    {
        $this->trigger = $trigger;
        $this->animationVar = $animationVar;
        $this->innerAnimationCB = $innerAnimationCB;
    }


    /**
     * Generates JavaScript code to attach a animation timeline trigger to the window.
     *
     * This function outputs an array of JS code lines that:
     *   - Select the window element
     *   - Attach an event listener
     *   - Execute a Animation timeline action and optional inner animation
     *
     * @return array Array of JavaScript code lines.
     * @since 1.1.0
     */
    public function generate(): array
    {
        $lines = [];
        $event = $this->trigger->getType();
        $action = $this->trigger->getAction();
        $delay = $this->trigger->getOptions()->getDelayInMilliseconds();

        $lines[] = "window.addEventListener('{$event}', function(e) {";
        $lines[] = "setTimeout(function() {";
        $lines[] = "\t{$this->animationVar}.{$action}();";

        $lines[] = "\t}, $delay);";
        $lines[] = "});";

        return $lines;
    }
}
