<?php

namespace FrontisInteraction\Generator\Triggers;

use FrontisInteraction\Models\Triggers\HoverTrigger;

defined('ABSPATH') || exit;

/**
 * Class to generate JavaScript code for a animation timeline trigger from a given HoverTrigger object.
 *
 * This class handles the user hover event options data.
 *
 * FrontisInteraction\Generator\Triggers
 */
class HoverEventGenerator implements TriggerEventGenerator
{
    /**
     * The HoverTrigger object to generate code for.
     *
     * @var HoverTrigger
     */
    private HoverTrigger $trigger;

    /**
     * The animation timeline variable to use in the generated code.
     *
     * @var string
     */
    private string $animationVar;

    /**
     * A callback function for generating triggered animation code.
     *
     * @var \Closure
     */
    private \Closure $innerAnimationCB;



    /**
     * Constructs a HoverEventGenerator object.
     *
     * Initializes the class with the given HoverTrigger object, animation timeline variable, and a callback for generating triggered animation code.
     *
     * @param HoverTrigger $trigger The HoverTrigger object to generate code for.
     * @param string $animationVar The animation timeline variable to use in the generated code.
     * @param \Closure $innerAnimationCB A callback function for generating triggered animation code.
     * @since 2.0.0
     */
    public function __construct(HoverTrigger $trigger, string $animationVar, \Closure $innerAnimationCB)
    {
        $this->trigger = $trigger;
        $this->animationVar = $animationVar;
        $this->innerAnimationCB = $innerAnimationCB;
    }


    /**
     * Generates JavaScript code to attach a animation timeline trigger to DOM elements.
     *
     * This function outputs an array of JS code lines that:
     *   - Select elements by a CSS selector
     *   - Attach an event listener
     *   - Execute a animation timeline action and optional animations
     *
     * @return array Array of JavaScript code lines.
     */
    public function generate(): array
    {
        $lines = [];
        $selector = $this->trigger->getTarget()->getSelector();
        $event = $this->trigger->getOptions()->getType();
        $action = $this->trigger->getAction();

        $lines[] = "document.querySelectorAll({$selector}).forEach(function(el) {";
        $lines[] = "\tel.addEventListener('{$event}', function(e) {";

        if ($this->trigger->getOptions()->isStopPropagation())
            $lines[] = "\t\te.stopPropagation();";

        $lines[] = "\t\t{$this->animationVar}.{$action}();";

        $lines[] = "\t});";
        $lines[] = "});";

        return $lines;
    }
}
