<?php

namespace FrontisInteraction\Generator;

use FrontisInteraction\Generator\Factories\TriggerGenerator;
use FrontisInteraction\Models\Interaction;
use FrontisInteraction\Models\Sequence\Sequence;
use FrontisInteraction\Models\TriggerPoint;

/**
 * Class to generate JavaScript code for animation timeline trigger from given interaction object
 * 
 * @package FrontisInteraction
 */
class MotionJSGenerator
{
    /**
     * The interaction data to generate code for.
     * 
     * @var array
     */
    private $data = [];

    /**
     * Initializes the MotionJSGenerator class with the given interaction data.
     *
     * @param array $data The interaction data to generate code for.
     */
    public function __construct(array $data)
    {
        $this->data = $data;
    }

    /**
     * Generate the JavaScript code for the given interactions.
     * 
     * The generated code includes the animation code, trigger code and breakpoint code.
     * The animation code defines the animation variables and executes the animation.
     * The trigger code defines the trigger event listeners and executes the animation.
     * The breakpoint code wraps the animation and trigger code with a breakpoint.
     * 
     * @return string The generated JavaScript code.
     */
    public function getContent(): string
    {
        $lines = [];

        foreach ($this->data as $interaction) {
            $ia = new Interaction($interaction);
            if (!$ia->isActive()) continue;

            // Animation code generate
            $animationVars = $ia->getId();
            $animationLines[] = "let $animationVars = Motion.animate([";
            foreach ($ia->getSequences() as $sequence) {
                $animationLines[] = "\t" . $this->getSequence($sequence) . ",";
            }
            $animationLines[] = "]);";
            $animationLines[] = $animationVars . ".pause();";

            // Trigger code generate
            $triggersLines = [];
            foreach ($ia->getTriggers() as $trigger) {
                $createTrigger = TriggerGenerator::create($trigger, $animationVars, fn() => $triggersLines);
                $triggerLines = $createTrigger->generate();
                $triggersLines = array_merge($triggersLines, $triggerLines);
            }

            $mergedLines = array_merge($animationLines, $triggersLines);

            $wrapWithBreakPoint = $this->generateBreakpoint($ia->getTriggerPoint(), $mergedLines);

            $lines = array_merge($lines, $wrapWithBreakPoint);
        }

        $wrappedWithReadyEvent = $this->onReadyEvent($lines);
        $utilsGenerate = $this->generateUtilsScripts();
        $allLines = array_merge($utilsGenerate, $wrappedWithReadyEvent);
        return implode("\n", $allLines) . "\n";
    }


    /**
     * Returns a string representing a sequence in MotionJS format.
     *
     * @param Sequence $sequence The sequence to be converted.
     *
     * @return string A string representing the sequence in MotionJS format.
     */
    public function getSequence(Sequence $sequence)
    {
        $element = $sequence->getTarget()->getSelector();
        $animatedProperties = $sequence->getProperties();
        $options = $sequence->getOptions()->toArray();

        return sprintf("[%s, %s, %s]", $element, json_encode($animatedProperties), json_encode($options));
    }


    /**
     * Generates a conditional wrapper around the given code lines.
     * The wrapper checks if the current breakpoint matches the given trigger point.
     * If it does, the code lines inside the wrapper are executed.
     *
     * @param TriggerPoint $triggerPoint The trigger point to check against the current breakpoint.
     * @param array $innerLines The code lines to wrap inside the conditional statement.
     *
     * @return array The generated code lines.
     * @since 1.1.0
     */
    public function generateBreakpoint(TriggerPoint $triggerPoint, array $innerLines): array
    {
        if ($triggerPoint->isEnabledInAllScreen())
            return $innerLines;

        $lines = [];
        $lines[] = "if (getBreakpoint({$triggerPoint->toJson()})){;";

        $lines = array_merge($lines, array_map(fn($line) => "\t{$line}", $innerLines));

        $lines[] = "};";

        return $lines;
    }


    /**
     * Generates JavaScript lines to attach an event listener for the 'DOMContentLoaded' event.
     * The listener target is chosen based on the event type: document.
     * The provided timeline lines will be inserted inside the event callback.
     *
     * @param array $innerLines The code lines to inject inside the event callback.
     *
     * @return array An array of JavaScript code lines as strings.
     * @since 1.1.0
     */
    public function onReadyEvent(array $innerLines = []): array
    {
        $lines = [];
        $lines[] = "document.addEventListener('DOMContentLoaded', function() {";

        $lines = array_merge($lines, array_map(fn($line) => "\t{$line}", $innerLines));

        $lines[] = "});";

        return $lines;
    }


    /**
     * Generates utility scripts for animation.
     * 
     * Currently, this method generates a single function, `getBreakpoint`, which takes an object of breakpoints
     * and returns the current breakpoint based on the window's inner width.
     * 
     * @return array An array of JavaScript code lines as strings.
     * @since 1.1.0
     */
    public function generateUtilsScripts(): array
    {
        $lines = [];

        $lines[] = "function getBreakpoint(breakpoints) {";
        $lines[] = "    const width = window.innerWidth;";
        $lines[] = "    if (width >= 1025) return breakpoints.desktop;";
        $lines[] = "    else if (width >= 769) return breakpoints.tablet;";
        $lines[] = "    else if (width >= 481) return breakpoints.mobileLandscape;";
        $lines[] = "    else return breakpoints.mobilePortrait;";
        $lines[] = "}";

        return $lines;
    }
}
