<?php

namespace FrontisInteraction;

use WP_Error;

defined('ABSPATH') || exit;

/**
 * Class responsible for writing animation js code to a file.
 *
 * @package FrontisInteraction
 */
class AnimationJSFileWriter
{
    private int $post_id;
    private string $content;
    private int $timestamp;

    private string $base_dir;
    private string $base_url;

    const META_KEY = '_fa_animation_file_meta';

    /**
     * Construct a new AnimationJSFileWriter.
     *
     * @param int    $post_id  The ID of the post to write the animation file for.
     * @param string $content   The animation content to write to a file.
     * @param int    $timestamp The timestamp of the animation file.
     */
    public function __construct(int $post_id, string $content, int $timestamp)
    {
        $this->post_id = $post_id;
        $this->content = $content;
        $this->timestamp = $timestamp;

        $upload_dir = wp_upload_dir();
        $this->base_dir = trailingslashit($upload_dir['basedir']) . 'frontis-interaction';
        $this->base_url = trailingslashit($upload_dir['baseurl']) . 'frontis-interaction';

        if (!file_exists($this->base_dir)) {
            wp_mkdir_p($this->base_dir);
        }
    }

    /**
     * Writes the animation js code to a file and saves the file's meta data.
     *
     * @return array|WP_Error Array with the file's URL, filename, and meta data on success, or WP_Error on failure.
     */
    public function write()
    {
        $slug = get_post_field('post_name', $this->post_id);
        $folderName = (string) $this->post_id;
        $filename = sprintf('%s-%s.js', $slug, $this->timestamp);

        // Remove any old files for this post
        $this->removeOldFile($folderName, $slug);

        $res = $this->writeFile($folderName, $filename, $this->content);
        if (is_wp_error($res)) {
            return $res;
        }

        $meta = [
            'filename'     => $filename,
            'url'          => $res['url'],
            'hash'         => $this->timestamp,
            'plugins'      => [],
            'generated_at' => current_time('mysql'),
        ];

        update_post_meta($this->post_id, self::META_KEY, $meta);

        return [
            'url'      => $res['url'],
            'filename' => $filename,
            'meta'     => $meta,
        ];
    }

    /**
     * Writes a file to the filesystem.
     *
     * This function takes care of creating the folder if it doesn't exist.
     * If the file cannot be written, it will return a WP_Error object.
     *
     * @param string $foldername The name of the folder to write the file to.
     * @param string $filename The name of the file to write.
     * @param string $content The contents of the file to write.
     *
     * @return array An array containing the file path and URL.
     */
    private function writeFile(string $foldername, string $filename, string $content)
    {
        $path_dir = trailingslashit($this->base_dir) . trailingslashit($foldername);
        $url_dir  = trailingslashit($this->base_url) . trailingslashit($foldername);

        if (!is_dir($path_dir)) {
            wp_mkdir_p($path_dir);
        }

        $file_path = $path_dir . $filename;
        $file_url  = $url_dir . $filename;

        $written = @file_put_contents($file_path, $content, LOCK_EX);

        if ($written === false) {
            if (!function_exists('request_filesystem_credentials')) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }

            $creds = request_filesystem_credentials(site_url() . '/wp-admin/', '', false, false, null);
            if ($creds === false) {
                return new WP_Error('fs_credentials', 'Unable to obtain filesystem credentials.');
            }

            global $wp_filesystem;
            if (empty($wp_filesystem) && !WP_Filesystem($creds)) {
                return new WP_Error('fs_error', 'WP_Filesystem initialization failed.');
            }

            if (!$wp_filesystem->put_contents($file_path, $content, FS_CHMOD_FILE)) {
                return new WP_Error('fs_write_failed', 'WP_Filesystem failed to write file.');
            }
        }

        return [
            'file' => $file_path,
            'url'  => $file_url,
        ];
    }

    /**
     * Remove old animation files for a given post.
     *
     * Files are searched in the format of `<post-slug>-<timestamp>.js` in
     * the folder for the post ID.
     *
     * @param string $foldername The folder name for the post ID.
     * @param string $slug The slug of the post.
     */
    private function removeOldFile(string $foldername, string $slug)
    {
        $pattern = trailingslashit($this->base_dir) . trailingslashit($foldername) . "{$slug}-*.js";
        foreach (glob($pattern) as $file) {
            @wp_delete_file($file);
        }
    }
}
