<?php

namespace FrontisInteraction;

use FrontisInteraction\Generator\MotionJSGenerator;

defined('ABSPATH') || exit;

/**
 * Class responsible for generating animation resources.
 *
 * @package FrontisInteraction
 */
class AnimationBuilder
{
    /** 
     * The one true AnimationBuilder
     * 
     * @var AnimationBuilder
     **/
    private static $instance;


    /**
     * Returns an instance of the AnimationBuilder class, or create one if none exist yet.
     * 
     * @return AnimationBuilder
     */
    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }


    /**
     * Hook into WordPress events to generate animation resources.
     * 
     * Hooks into `added_post_meta`, `update_post_meta`, `wp_enqueue_scripts`, and `before_delete_post` 
     * to generate animation resources when animation data is added, updated, or deleted.
     * 
     * @return void
     */
    private function __construct()
    {
        add_action('added_post_meta', array($this, 'generate_animation_resources'), 10, 4);
        add_action('update_post_meta', array($this, 'generate_animation_resources'), 10, 4);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_animation_resources'], 100);
        add_action('before_delete_post', [$this, 'delete_animation_resources'], 10);
    }


    /**
     * Generate animation resources when animation data is added, updated, or deleted.
     *
     * Hooked into `added_post_meta`, `update_post_meta`, and `before_delete_post` to generate animation resources
     * when animation data is added, updated, or deleted.
     *
     * @param int $meta_id The ID of the meta field being added/updated/deleted.
     * @param int $post_id The ID of the post being modified.
     * @param string $meta_key The key of the meta field being added/updated/deleted.
     * @param string $meta_value The value of the meta field being added/updated/deleted.
     *
     * @return void
     */
    public function generate_animation_resources($meta_id, $post_id, $meta_key, $meta_value)
    {
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) return;

        // Only execute if animation data is available
        if ($meta_key !== '_frontisInteractions')  return;

        $data = json_decode($meta_value, true);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) return;

        $timestamp = get_post_meta($post_id, '_interactionEditorTimestamp', true);
        if (!$timestamp) $timestamp = time();

        $generated_file_meta = get_post_meta($post_id, AnimationJSFileWriter::META_KEY, true);
        // Only execute if animation data has changed
        if ($generated_file_meta && isset($generated_file_meta['timestamp']) && $generated_file_meta['timestamp'] === $timestamp) return;

        $generator = new MotionJSGenerator($data);
        $JSContent = apply_filters('frontis_interaction_animation_js', $generator->getContent(), $data, $post_id);

        $writer = new AnimationJSFileWriter($post_id, $JSContent, $timestamp);
        $writer->write();
    }

    /**
     * Enqueues animation resources for a post when it is rendered.
     * 
     * This function is hooked into `wp_enqueue_scripts` and is responsible for enqueuing the animation
     * resources generated by the `generate_animation_resources` method.
     * 
     * @return void
     */
    public function enqueue_animation_resources()
    {
        global $post;
        if (empty($post->ID)) return;

        $meta = get_post_meta($post->ID, AnimationJSFileWriter::META_KEY, true);

        if (empty($meta) || empty($meta['url'])) return;

        $url = $meta['url'];
        $handle = 'frontis-interaction-' . $post->ID;

        // Determine a version
        $version = $meta['version'] ?? ($meta['mtime'] ?? null) ?? md5($url);

        wp_enqueue_script($handle, $url, ['motion'], $version, true);
    }


    /**
     * Deletes animation resources when a post is deleted.
     * 
     * This function is hooked into `before_delete_post` and is responsible for deleting the animation resources
     * generated by the `generate_animation_resources` method.
     * 
     * @param int $post_id The ID of the post being deleted.
     * 
     * @return void
     */
    public function delete_animation_resources($post_id)
    {
        $meta = get_post_meta($post_id, AnimationJSFileWriter::META_KEY, true);

        if (empty($meta['url'])) return;

        $upload_dir = wp_upload_dir();
        $file_path  = str_replace($upload_dir['baseurl'], $upload_dir['basedir'], $meta['url']);

        if (! function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        global $wp_filesystem;
        WP_Filesystem();

        if ($wp_filesystem->exists($file_path)) {
            $wp_filesystem->delete($file_path, true);
        }

        $folder_path = dirname($file_path);

        $base_dir = trailingslashit($upload_dir['basedir']) . 'frontis-interaction';
        if (strpos($folder_path, $base_dir) === 0 && $wp_filesystem->is_dir($folder_path)) {

            $files = $wp_filesystem->dirlist($folder_path);
            if (empty($files)) {
                $wp_filesystem->rmdir($folder_path);
            }
        }
    }
}
