<?php
/**
 * Theme installation tasks class
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Fresh_Install_Assistant_Theme_Tasks {
    
    /**
     * Available themes for installation
     */
    private $themes = array(
        'astra' => array(
            'name' => 'Astra',
            'slug' => 'astra'
        ),
        'generatepress' => array(
            'name' => 'GeneratePress',
            'slug' => 'generatepress'
        ),
        'oceanwp' => array(
            'name' => 'OceanWP',
            'slug' => 'oceanwp'
        ),
        'blocksy' => array(
            'name' => 'Blocksy',
            'slug' => 'blocksy'
        ),
        'neve' => array(
            'name' => 'Neve',
            'slug' => 'neve'
        ),
        'hello-elementor' => array(
            'name' => 'Hello Elementor',
            'slug' => 'hello-elementor'
        ),
        'kadence' => array(
            'name' => 'Kadence',
            'slug' => 'kadence'
        )
    );
    
    /**
     * Install and activate a theme
     */
    public function install_theme($theme_key) {
        if (!isset($this->themes[$theme_key])) {
            return array(
                'status' => 'failed',
                'message' => __('Invalid theme selected', 'fresh-install-assistant')
            );
        }
        
        $theme = $this->themes[$theme_key];
        $theme_slug = $theme['slug'];
        
        // Check if theme is already installed
        $installed_theme = wp_get_theme($theme_slug);
        if ($installed_theme->exists()) {
            // Theme is installed, just activate it
            if (get_option('stylesheet') === $theme_slug) {
                return array(
                    'status' => 'skipped',
                    'message' => sprintf(
                        /* translators: %s: theme name */
                        __('%s theme is already active', 'fresh-install-assistant'), 
                        $theme['name']
                    )
                );
            }
            
            switch_theme($theme_slug);
            return array(
                'status' => 'completed',
                'message' => sprintf(
                    /* translators: %s: theme name */
                    __('%s theme activated', 'fresh-install-assistant'), 
                    $theme['name']
                )
            );
        }
        
        // Install the theme - Load required admin files only when needed
        if (!function_exists('themes_api')) {
            require_once ABSPATH . 'wp-admin/includes/theme.php';
        }
        if (!class_exists('Theme_Upgrader')) {
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        }
        if (!function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        if (!function_exists('wp_print_admin_notice_templates')) {
            require_once ABSPATH . 'wp-admin/includes/misc.php';
        }
        
        try {
            // Capture output to prevent display during AJAX
            ob_start();
            
            // Use the Automatic_Upgrader_Skin for silent operation
            $skin = new Automatic_Upgrader_Skin();
            $upgrader = new Theme_Upgrader($skin);
            
            // Use WordPress theme API - Load only if needed
            if (!function_exists('themes_api')) {
                require_once ABSPATH . 'wp-admin/includes/theme-install.php';
            }
            
            $api = themes_api('theme_information', array(
                'slug' => $theme_slug,
                'fields' => array(
                    'sections' => false,
                    'tags' => false
                )
            ));
            
            if (is_wp_error($api)) {
                ob_end_clean(); // Clean up output buffer
                return array(
                    'status' => 'failed',
                    'message' => sprintf(
                        /* translators: %s: error message */
                        __('Failed to get theme information: %s', 'fresh-install-assistant'), 
                        $api->get_error_message()
                    )
                );
            }
            
            if (!isset($api->download_link)) {
                ob_end_clean(); // Clean up output buffer
                return array(
                    'status' => 'failed',
                    'message' => __('Failed to retrieve theme download link', 'fresh-install-assistant')
                );
            }
            
            // Install the theme
            $result = $upgrader->install($api->download_link);
            
            // Clean up output buffer
            ob_end_clean();
            
            if (is_wp_error($result)) {
                return array(
                    'status' => 'failed',
                    'message' => sprintf(
                        /* translators: %s: error message */
                        __('Failed to install theme: %s', 'fresh-install-assistant'), 
                        $result->get_error_message()
                    )
                );
            }
            
            if (!$result) {
                return array(
                    'status' => 'failed',
                    'message' => __('Theme installation failed', 'fresh-install-assistant')
                );
            }
            
            // Activate the theme
            switch_theme($theme_slug);
        } catch (Exception $e) {
            // Make sure to clean up output buffer if an exception occurs
            if (ob_get_level()) {
                ob_end_clean();
            }
            
            return array(
                'status' => 'failed',
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error installing theme: %s', 'fresh-install-assistant'),
                    $e->getMessage()
                )
            );
        }
        
        return array(
            'status' => 'completed',
            'message' => sprintf(
                /* translators: %s: theme name */
                __('%s theme installed and activated', 'fresh-install-assistant'), 
                $theme['name']
            )
        );
    }
}