<?php
/**
 * Setup tasks class for creating pages and menus
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Fresh_Install_Assistant_Setup_Tasks {
    
    /**
     * Task data passed from handler
     */
    private $task_data = array();
    
    /**
     * Constructor
     */
    public function __construct($task_data = array()) {
        $this->task_data = $task_data;
    }
    
    /**
     * Pages to create
     */
    private $pages = array(
        'home' => array(
            'title' => 'Home',
            'content' => ''
        ),
        'services' => array(
            'title' => 'Services',
            'content' => ''
        ),
        'about' => array(
            'title' => 'About',
            'content' => ''
        ),
        'contact' => array(
            'title' => 'Contact',
            'content' => ''
        ),
        'blog' => array(
            'title' => 'Blog',
            'content' => ''
        )
    );
    
    /**
     * Set site title and tagline
     */
    public function task_set_site_info() {
        // Get the values from task data
        $site_title = isset($this->task_data['site_title']) ? $this->task_data['site_title'] : '';
        $site_tagline = isset($this->task_data['site_tagline']) ? $this->task_data['site_tagline'] : '';
        
        // Check if we have values to update
        if (empty($site_title) && empty($site_tagline)) {
            return array(
                'status' => 'skipped',
                'message' => __('No site title or tagline provided', 'fresh-install-assistant')
            );
        }
        
        $updated = array();
        
        // Update site title if provided
        if (!empty($site_title)) {
            $current_title = get_option('blogname');
            if ($current_title !== $site_title) {
                update_option('blogname', $site_title);
                $updated[] = __('Site title', 'fresh-install-assistant');
            }
        }
        
        // Update tagline if provided
        if (!empty($site_tagline)) {
            $current_tagline = get_option('blogdescription');
            if ($current_tagline !== $site_tagline) {
                update_option('blogdescription', $site_tagline);
                $updated[] = __('Tagline', 'fresh-install-assistant');
            }
        }
        
        if (empty($updated)) {
            return array(
                'status' => 'skipped',
                'message' => __('Site title and tagline already set to provided values', 'fresh-install-assistant')
            );
        }
        
        return array(
            'status' => 'completed',
            'message' => sprintf(
                /* translators: %s: list of updated items */
                __('%s updated successfully', 'fresh-install-assistant'), 
                implode(' and ', $updated)
            )
        );
    }
    
    /**
     * Create pages
     */
    public function task_create_pages() {
        // Get selected pages from task data
        $pages_to_create = isset($this->task_data['pages_to_create']) ? $this->task_data['pages_to_create'] : array_keys($this->pages);
        
        // If no pages selected, skip
        if (empty($pages_to_create)) {
            return array(
                'status' => 'skipped',
                'message' => __('No pages selected for creation', 'fresh-install-assistant')
            );
        }
        
        $created = 0;
        $skipped = 0;
        $messages = array();
        
        foreach ($pages_to_create as $slug) {
            // Check if this page type exists in our pages array
            if (!isset($this->pages[$slug])) {
                continue;
            }
            
            $page_data = $this->pages[$slug];
            
            // Check if page already exists
            $existing = get_page_by_path($slug);
            
            if ($existing) {
                $skipped++;
                $messages[] = sprintf(
                    /* translators: %s: page title */
                    __('Page "%s" already exists', 'fresh-install-assistant'), 
                    $page_data['title']
                );
                continue;
            }
            
            // Create the page
            $page_id = wp_insert_post(array(
                'post_title' => $page_data['title'],
                'post_content' => $page_data['content'],
                'post_status' => 'publish',
                'post_type' => 'page',
                'post_name' => $slug
            ));
            
            if (!is_wp_error($page_id)) {
                $created++;
            } else {
                $messages[] = sprintf(
                    /* translators: %s: page title */
                    __('Failed to create "%s" page', 'fresh-install-assistant'), 
                    $page_data['title']
                );
            }
        }
        
        if ($created === 0 && $skipped === count($pages_to_create)) {
            return array(
                'status' => 'skipped',
                'message' => __('All selected pages already exist', 'fresh-install-assistant')
            );
        }
        
        return array(
            'status' => 'completed',
            'message' => sprintf(
                /* translators: 1: number of pages created, 2: number of pages skipped */
                __('Pages created: %1$d, Skipped: %2$d', 'fresh-install-assistant'), 
                $created, 
                $skipped
            ) . (!empty($messages) ? ' (' . implode(', ', $messages) . ')' : '')
        );
    }
    
    /**
     * Set home page as front page
     */
    public function task_set_front_page() {
        // Get the home page
        $home_page = get_page_by_path('home');
        
        if (!$home_page) {
            return array(
                'status' => 'failed',
                'message' => __('Home page not found. Please create pages first.', 'fresh-install-assistant')
            );
        }
        
        // Check if already set
        if (get_option('show_on_front') === 'page' && 
            get_option('page_on_front') == $home_page->ID) {
            return array(
                'status' => 'skipped',
                'message' => __('Home page already set as front page', 'fresh-install-assistant')
            );
        }
        
        // Set the front page
        update_option('show_on_front', 'page');
        update_option('page_on_front', $home_page->ID);
        
        return array(
            'status' => 'completed',
            'message' => __('Home page set as front page', 'fresh-install-assistant')
        );
    }
    
    /**
     * Set Blog page as posts page
     */
    public function task_set_blog_page() {
        // Get the blog page
        $blog_page = get_page_by_path('blog');
        
        if (!$blog_page) {
            return array(
                'status' => 'failed',
                'message' => __('Blog page not found. Please create the Blog page first.', 'fresh-install-assistant')
            );
        }
        
        // Check if already set
        if (get_option('page_for_posts') == $blog_page->ID) {
            return array(
                'status' => 'skipped',
                'message' => __('Blog page already set as posts page', 'fresh-install-assistant')
            );
        }
        
        // Set the blog page as posts page
        update_option('page_for_posts', $blog_page->ID);
        
        return array(
            'status' => 'completed',
            'message' => __('Blog page set as posts page', 'fresh-install-assistant')
        );
    }
    
    /**
     * Create main navigation menu
     */
    public function task_create_menu() {
        $menu_name = 'Main Menu';
        $menu_exists = wp_get_nav_menu_object($menu_name);
        
        if ($menu_exists) {
            return array(
                'status' => 'skipped',
                'message' => __('Main Menu already exists', 'fresh-install-assistant')
            );
        }
        
        // Create the menu
        $menu_id = wp_create_nav_menu($menu_name);
        
        if (is_wp_error($menu_id)) {
            return array(
                'status' => 'failed',
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Failed to create menu: %s', 'fresh-install-assistant'), 
                    $menu_id->get_error_message()
                )
            );
        }
        
        // Add pages to menu
        $menu_items_added = 0;
        // Get the pages that were created from task data
        $pages_to_create = isset($this->task_data['pages_to_create']) ? $this->task_data['pages_to_create'] : array('home', 'services', 'about', 'contact', 'blog');
        $page_order = array('home', 'services', 'about', 'contact', 'blog');
        
        foreach ($page_order as $position => $slug) {
            // Only add to menu if it was selected for creation
            if (!in_array($slug, $pages_to_create)) {
                continue;
            }
            
            $page = get_page_by_path($slug);
            
            if ($page) {
                $item_data = array(
                    'menu-item-object-id' => $page->ID,
                    'menu-item-object' => 'page',
                    'menu-item-type' => 'post_type',
                    'menu-item-status' => 'publish',
                    'menu-item-position' => $position + 1
                );
                
                $result = wp_update_nav_menu_item($menu_id, 0, $item_data);
                
                if (!is_wp_error($result)) {
                    $menu_items_added++;
                }
            }
        }
        
        // Set menu location if theme supports it
        $locations = get_theme_mod('nav_menu_locations');
        if (!is_array($locations)) {
            $locations = array();
        }
        
        // Try common menu location names
        $possible_locations = array('primary', 'main', 'header', 'top');
        $registered_locations = get_registered_nav_menus();
        
        foreach ($possible_locations as $location) {
            if (isset($registered_locations[$location])) {
                $locations[$location] = $menu_id;
                break;
            }
        }
        
        set_theme_mod('nav_menu_locations', $locations);
        
        return array(
            'status' => 'completed',
            'message' => sprintf(
                /* translators: %d: number of menu items added */
                __('Main Menu created with %d items', 'fresh-install-assistant'), 
                $menu_items_added
            )
        );
    }
}