<?php
/**
 * Cleanup tasks class
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Fresh_Install_Assistant_Cleanup_Tasks {
    
    /**
     * Remove Hello World post
     */
    public function task_remove_hello_world() {
        $post = get_page_by_path('hello-world', OBJECT, 'post');
        
        if (!$post) {
            return array(
                'status' => 'skipped',
                'message' => __('Hello World post not found', 'fresh-install-assistant')
            );
        }
        
        $result = wp_delete_post($post->ID, true);
        
        return array(
            'status' => $result ? 'completed' : 'failed',
            'message' => $result 
                ? __('Hello World post removed', 'fresh-install-assistant')
                : __('Failed to remove Hello World post', 'fresh-install-assistant')
        );
    }
    
    /**
     * Remove Sample Page
     */
    public function task_remove_sample_page() {
        $page = get_page_by_path('sample-page', OBJECT, 'page');
        
        if (!$page) {
            return array(
                'status' => 'skipped',
                'message' => __('Sample Page not found', 'fresh-install-assistant')
            );
        }
        
        $result = wp_delete_post($page->ID, true);
        
        return array(
            'status' => $result ? 'completed' : 'failed',
            'message' => $result 
                ? __('Sample Page removed', 'fresh-install-assistant')
                : __('Failed to remove Sample Page', 'fresh-install-assistant')
        );
    }
    
    /**
     * Rename Uncategorized to Blog
     */
    public function task_rename_uncategorized() {
        $cat = get_category_by_slug('uncategorized');
        
        if (!$cat) {
            return array(
                'status' => 'skipped',
                'message' => __('Uncategorized category not found', 'fresh-install-assistant')
            );
        }
        
        // Check if already renamed
        if ($cat->name === 'Blog') {
            return array(
                'status' => 'skipped',
                'message' => __('Category already renamed to Blog', 'fresh-install-assistant')
            );
        }
        
        $result = wp_update_term($cat->term_id, 'category', array(
            'name' => 'Blog',
            'slug' => 'blog'
        ));
        
        return array(
            'status' => is_wp_error($result) ? 'failed' : 'completed',
            'message' => is_wp_error($result) 
                ? sprintf(
                    /* translators: %s: error message */
                    __('Failed to rename category: %s', 'fresh-install-assistant'), 
                    $result->get_error_message()
                )
                : __('Uncategorized renamed to Blog', 'fresh-install-assistant')
        );
    }
    
    /**
     * Set permalink structure
     */
    public function task_set_permalinks() {
        $current_structure = get_option('permalink_structure');
        
        if ($current_structure === '/%postname%/') {
            return array(
                'status' => 'skipped',
                'message' => __('Permalink structure already set correctly', 'fresh-install-assistant')
            );
        }
        
        global $wp_rewrite;
        $wp_rewrite->set_permalink_structure('/%postname%/');
        $wp_rewrite->flush_rules();
        
        return array(
            'status' => 'completed',
            'message' => __('Permalink structure set to /%postname%/', 'fresh-install-assistant')
        );
    }
    
    /**
     * Remove Hello Dolly plugin
     */
    public function task_remove_hello_dolly() {
        $plugin_file = 'hello-dolly/hello.php';
        
        if (!file_exists(WP_PLUGIN_DIR . '/hello-dolly/hello.php')) {
            return array(
                'status' => 'skipped',
                'message' => __('Hello Dolly plugin not found', 'fresh-install-assistant')
            );
        }
        
        try {
            // Deactivate if active
            if (is_plugin_active($plugin_file)) {
                deactivate_plugins($plugin_file);
            }
            
            // Include required files for plugin deletion
            if (!function_exists('delete_plugins')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            
            // Delete the plugin
            $result = delete_plugins(array($plugin_file));
            
            return array(
                'status' => is_wp_error($result) ? 'failed' : 'completed',
                'message' => is_wp_error($result)
                    ? sprintf(
                        /* translators: %s: error message */
                        __('Failed to remove Hello Dolly: %s', 'fresh-install-assistant'), 
                        $result->get_error_message()
                    )
                    : __('Hello Dolly plugin removed', 'fresh-install-assistant')
            );
        } catch (Exception $e) {
            return array(
                'status' => 'failed',
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error removing Hello Dolly: %s', 'fresh-install-assistant'),
                    $e->getMessage()
                )
            );
        }
    }
    
    /**
     * Remove Akismet plugin
     */
    public function task_remove_akismet() {
        $plugin_file = 'akismet/akismet.php';
        
        if (!file_exists(WP_PLUGIN_DIR . '/akismet/akismet.php')) {
            return array(
                'status' => 'skipped',
                'message' => __('Akismet plugin not found', 'fresh-install-assistant')
            );
        }
        
        try {
            // Deactivate if active
            if (is_plugin_active($plugin_file)) {
                deactivate_plugins($plugin_file);
            }
            
            // Include required files for plugin deletion
            if (!function_exists('delete_plugins')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            
            // Delete the plugin
            $result = delete_plugins(array($plugin_file));
            
            return array(
                'status' => is_wp_error($result) ? 'failed' : 'completed',
                'message' => is_wp_error($result)
                    ? sprintf(
                        /* translators: %s: error message */
                        __('Failed to remove Akismet: %s', 'fresh-install-assistant'), 
                        $result->get_error_message()
                    )
                    : __('Akismet plugin removed', 'fresh-install-assistant')
            );
        } catch (Exception $e) {
            return array(
                'status' => 'failed',
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error removing Akismet: %s', 'fresh-install-assistant'),
                    $e->getMessage()
                )
            );
        }
    }
    
    /**
     * Disable comments and pingbacks
     */
    public function task_disable_comments() {
        // Check if already disabled
        if (get_option('default_comment_status') === 'closed' && 
            get_option('default_ping_status') === 'closed') {
            return array(
                'status' => 'skipped',
                'message' => __('Comments and pingbacks already disabled', 'fresh-install-assistant')
            );
        }
        
        try {
            // Disable comments on future posts
            update_option('default_comment_status', 'closed');
            update_option('default_ping_status', 'closed');
            
            // Disable comments on existing posts - batch process to avoid timeout
            $batch_size = 100;
            $offset = 0;
            $total_processed = 0;
            
            do {
                $posts = get_posts(array(
                    'post_type' => array('post', 'page'),
                    'post_status' => 'any',
                    'numberposts' => $batch_size,
                    'offset' => $offset,
                    'fields' => 'ids'
                ));
                
                if (empty($posts)) {
                    break;
                }
                
                foreach ($posts as $post_id) {
                    wp_update_post(array(
                        'ID' => $post_id,
                        'comment_status' => 'closed',
                        'ping_status' => 'closed'
                    ));
                    $total_processed++;
                }
                
                $offset += $batch_size;
                
                // Prevent timeout on large sites
                if ($total_processed > 500) {
                    break;
                }
            } while (count($posts) === $batch_size);
            
            return array(
                'status' => 'completed',
                'message' => sprintf(
                    /* translators: %d: number of posts processed */
                    __('Comments and pingbacks disabled (%d posts processed)', 'fresh-install-assistant'),
                    $total_processed
                )
            );
        } catch (Exception $e) {
            return array(
                'status' => 'failed',
                'message' => sprintf(
                    /* translators: %s: error message */
                    __('Error disabling comments: %s', 'fresh-install-assistant'),
                    $e->getMessage()
                )
            );
        }
    }
    
    /**
     * Hide dashboard widgets
     */
    public function task_hide_dashboard_widgets() {
        // List of default dashboard widgets to hide
        $widgets_to_hide = array(
            'dashboard_right_now',      // At a Glance
            'dashboard_activity',       // Activity
            'dashboard_quick_press',    // Quick Draft
            'dashboard_primary',        // WordPress Events and News
            'dashboard_secondary',      // Other WordPress News (deprecated but might exist)
            'dashboard_site_health',    // Site Health Status
            'dashboard_recent_comments' // Recent Comments
        );
        
        // Get current hidden widgets
        $hidden_widgets = get_user_meta(get_current_user_id(), 'metaboxhidden_dashboard', true);
        if (!is_array($hidden_widgets)) {
            $hidden_widgets = array();
        }
        
        // Check if already hidden
        $already_hidden = true;
        foreach ($widgets_to_hide as $widget) {
            if (!in_array($widget, $hidden_widgets)) {
                $already_hidden = false;
                break;
            }
        }
        
        // Check if welcome panel is already hidden
        $welcome_panel_hidden = get_user_meta(get_current_user_id(), 'show_welcome_panel', true);
        if ($welcome_panel_hidden != '0') {
            $already_hidden = false;
        }
        
        if ($already_hidden) {
            return array(
                'status' => 'skipped',
                'message' => __('Dashboard widgets already hidden', 'fresh-install-assistant')
            );
        }
        
        // Merge and update dashboard widgets
        $hidden_widgets = array_unique(array_merge($hidden_widgets, $widgets_to_hide));
        update_user_meta(get_current_user_id(), 'metaboxhidden_dashboard', $hidden_widgets);
        
        // Hide welcome panel
        update_user_meta(get_current_user_id(), 'show_welcome_panel', 0);
        
        // Update for all users (not just administrators)
        $all_users = get_users();
        foreach ($all_users as $user) {
            update_user_meta($user->ID, 'metaboxhidden_dashboard', $hidden_widgets);
            update_user_meta($user->ID, 'show_welcome_panel', 0);
        }
        
        return array(
            'status' => 'completed',
            'message' => __('Dashboard widgets and welcome panel hidden for all users', 'fresh-install-assistant')
        );
    }
}